package ru.yandex.solomon.gateway.api.internal;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.v3.CreateFavoriteRequest;
import ru.yandex.monitoring.v3.DeleteFavoriteRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.conf.db3.EntityType;
import ru.yandex.solomon.conf.db3.ydb.FavoriteRecord;
import ru.yandex.solomon.gateway.api.internal.dto.FavoriteDto;
import ru.yandex.solomon.gateway.api.internal.dto.FavoritesDto;
import ru.yandex.solomon.gateway.api.v3alpha.MonitoringFavoriteService;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal/favoriteProjects", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class FavoriteProjectsController {
    private final MonitoringFavoriteService monitoringFavoriteService;

    public FavoriteProjectsController(MonitoringFavoriteService monitoringFavoriteService) {
        this.monitoringFavoriteService = monitoringFavoriteService;
    }

    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<FavoritesDto> listFavorites(@RequireAuth AuthSubject subject) {
        return monitoringFavoriteService.list(EntityType.ENTITY_TYPE_PROJECT, subject)
                .thenApply(favoriteRecords -> {
                    List<FavoriteDto> result = favoriteRecords.getFavoritesList().stream()
                            .map(FavoriteDto::fromModel)
                            .collect(Collectors.toList());
                    return new FavoritesDto(result);
                });
    }

    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<FavoriteDto> createFavorite(
            @RequireAuth AuthSubject subject,
            @RequestParam("projectId") String projectId)
    {
        FavoriteRecord record = new FavoriteRecord(subject.getUniqueId(), EntityType.ENTITY_TYPE_PROJECT, projectId);

        CreateFavoriteRequest request = CreateFavoriteRequest.newBuilder()
                .setType(ru.yandex.monitoring.v3.EntityType.ENTITY_TYPE_PROJECT)
                .setId(projectId)
                .build();

        return monitoringFavoriteService.create(request, subject)
                .thenApply(FavoriteDto::fromModel);
    }

    @RequestMapping(method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteFavorite(
            @RequireAuth AuthSubject subject,
            @RequestParam("projectId") String projectId)
    {
        DeleteFavoriteRequest request = DeleteFavoriteRequest.newBuilder()
                .setType(ru.yandex.monitoring.v3.EntityType.ENTITY_TYPE_PROJECT)
                .setId(projectId)
                .build();

        return monitoringFavoriteService.delete(request, subject).thenAccept(ignored -> {});
    }
}
