package ru.yandex.solomon.gateway.api.internal;

import java.io.IOException;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.core.conf.flags.FeatureFlagsHolderImpl;

/**
 * @author Alexey Trushkin
 */
@RestController
@RequestMapping(path = "/api/internal/feature-flags", produces = MediaType.APPLICATION_JSON_VALUE)
public class FeatureFlagController {

    private final Authorizer authorizer;
    private final FeatureFlagsHolderImpl holder;
    private final ObjectMapper objectMapper;
    private final boolean isIdm;

    public FeatureFlagController(
            Authorizer authorizer,
            FeatureFlagsHolderImpl holder,
            ObjectMapper objectMapper,
            Optional<TIdmConfig> config)
    {
        this.authorizer = authorizer;
        this.holder = holder;
        this.objectMapper = objectMapper;
        this.isIdm = config.isPresent();
    }

    @ApiOperation(value = "List project feature flags")
    @GetMapping(path = "/{projectId}")
    public CompletableFuture<String> featureFlags(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId) {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                .thenApply(account -> {
                    try {
                        var strOriginal =  objectMapper.writeValueAsString(holder.flags(projectId));
                        if (!isIdm) {
                            return strOriginal;
                        }
                        //TODO remove after MONITORING-4187
                        Map<String, Boolean> map = objectMapper.readValue(strOriginal, new TypeReference<Map<String, Boolean>>() {});
                        map.put("USE_IDM_AUTHORIZER_ONLY", true);
                        return objectMapper.writeValueAsString(map);
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                });
    }
}
