package ru.yandex.solomon.gateway.api.internal;

import java.util.concurrent.CompletableFuture;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.core.conf.ClustersManager;
import ru.yandex.solomon.core.conf.DashboardsManager;
import ru.yandex.solomon.core.conf.GraphsManager;
import ru.yandex.solomon.core.conf.ServicesManager;
import ru.yandex.solomon.core.conf.ShardsManager;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.gateway.api.v2.dto.GlobalEntityListItemDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.ShardListItemDto;
import ru.yandex.solomon.ydb.page.PageOptions;


/**
 * This is non public API. Must be available for administrators only.
 *
 * @author Sergey Polovko
 */
@RestController
@RequestMapping(path = "/api/internal", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ShardsManager.class })
public class GlobalConfController {

    @Autowired
    private ShardsManager shardsManager;
    @Autowired
    private ClustersManager clustersManager;
    @Autowired
    private ServicesManager servicesManager;
    @Autowired
    private GraphsManager graphsManager;
    @Autowired
    private DashboardsManager dashboardsManager;
    @Autowired
    private InternalAuthorizer authorizer;


    @RequestMapping(path = "/shards", method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<ShardListItemDto>> getAllShards(
        @RequireAuth AuthSubject subject,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "state", defaultValue = ShardState.DEFAULT_NAME) ShardState state,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject)
            .thenCompose(aVoid -> shardsManager.getAllShards(pageOptions, state, text))
            .thenApply(shards -> PagedResultDto.fromModel(shards, ShardListItemDto::fromModel));
    }

    @RequestMapping(path = "/clusters", method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<GlobalEntityListItemDto>> getAllClusters(
        @RequireAuth AuthSubject subject,
        @RequestParam(value = "text", defaultValue = "") String text,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject)
            .thenCompose(aVoid -> clustersManager.getAllClusters(pageOptions, text))
            .thenApply(clusters -> PagedResultDto.fromModel(clusters, GlobalEntityListItemDto::fromModel));
    }

    @RequestMapping(path = "/services", method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<GlobalEntityListItemDto>> getAllServices(
        @RequireAuth AuthSubject subject,
        @RequestParam(value = "text", defaultValue = "") String text,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject)
            .thenCompose(aVoid -> servicesManager.getAllServices(pageOptions, text))
            .thenApply(services -> PagedResultDto.fromModel(services, GlobalEntityListItemDto::fromModel));
    }

    @RequestMapping(path = "/graphs", method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<GlobalEntityListItemDto>> getAllGraphs(
        @RequireAuth AuthSubject subject,
        @RequestParam(value = "text", defaultValue = "") String text,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject)
            .thenCompose(aVoid -> graphsManager.getAllGraphs(pageOptions, text))
            .thenApply(graphs -> PagedResultDto.fromModel(graphs, GlobalEntityListItemDto::fromModel));
    }

    @RequestMapping(path = "/dashboards", method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<GlobalEntityListItemDto>> getAllDashboards(
        @RequireAuth AuthSubject subject,
        @RequestParam(value = "text", defaultValue = "") String text,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject)
            .thenCompose(aVoid -> dashboardsManager.getAllDashboards(pageOptions, text))
            .thenApply(dashboards -> PagedResultDto.fromModel(dashboards, GlobalEntityListItemDto::fromModel));
    }
}
