package ru.yandex.solomon.gateway.api.internal;

import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.gateway.AlertStatusRequest;
import ru.yandex.monitoring.gateway.AlertStatusResponse;
import ru.yandex.monitoring.gateway.QueryTopRequest;
import ru.yandex.monitoring.gateway.QueryTopResponse;
import ru.yandex.monitoring.gateway.ReadDataRequest;
import ru.yandex.monitoring.gateway.ReadDataResponse;
import ru.yandex.monitoring.gateway.YasmGatewayServiceGrpc;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;
import ru.yandex.solomon.gateway.api.internal.yasm.YasmGatewayService;
import ru.yandex.solomon.grpc.handler.GrpcMethodHandler;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Ivan Tsybulin
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcYasmGatewayService extends YasmGatewayServiceGrpc.YasmGatewayServiceImplBase implements GrpcService {
    private final YasmGatewayService service;

    @Autowired
    public GrpcYasmGatewayService(YasmGatewayService service) {
        this.service = service;
    }

    private static <ReqT, RespT> void handle(
            Function<ReqT, CompletableFuture<RespT>> handler,
            ReqT request,
            StreamObserver<RespT> responseObserver)
    {
        asyncComplete(
                CompletableFutures.safeCall(() -> handler.apply(request))
                        .exceptionally(t -> {
                            throw GrpcApiExceptionResolver.doResolveException(t);
                        }),
                responseObserver
        );
    }

    private static final String readDataName = "GrpcYasmGatewayService#readData";

    @Override
    public void readData(ReadDataRequest request, StreamObserver<ReadDataResponse> responseObserver) {
        GrpcMethodHandler.handle(service::readData, request, responseObserver);
    }

    private static final String queryTopName = "GrpcYasmGatewayService#queryTop";

    @Override
    public void queryTop(QueryTopRequest request, StreamObserver<QueryTopResponse> responseObserver) {
        handle(service::queryTop, request, responseObserver);
    }

    private static final String alertStatusName = "GrpcYasmGatewayService#alertStatus";

    @Override
    public void alertStatus(AlertStatusRequest request, StreamObserver<AlertStatusResponse> responseObserver) {
        handle(service::alertStatus, request, responseObserver);
    }
}
