package ru.yandex.solomon.gateway.api.internal;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.salmon.fetcher.proto.FetcherApiProto;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.fetcher.client.FetcherClient;
import ru.yandex.solomon.gateway.api.internal.dto.ResolveClusterResponseDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterDto;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal/fetcher", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class InternalFetcherController {
    private static final ObjectMapper jsonMapper = new ObjectMapper();

    private final FetcherClient fetcherClient;

    @Autowired
    public InternalFetcherController(FetcherClient fetcherClient) {
        this.fetcherClient = fetcherClient;
    }

    @RequestMapping(path = "/resolveCluster", method = RequestMethod.POST)
    CompletableFuture<ResolveClusterResponseDto> resolveCluster(
        @RequireAuth AuthSubject subject,
        @RequestBody ClusterDto body)
    {
        Cluster cluster = ClusterDto.toModel(body);
        FetcherApiProto.TResolveClusterRequest resolveClusterRequest = toResolveClusterRequest(cluster);
        String host = fetcherClient.getAnyAvailableHost();
        return fetcherClient.resolveCluster(host, resolveClusterRequest)
            .thenApply(ResolveClusterResponseDto::fromModel);
    }

    private static FetcherApiProto.TResolveClusterRequest toResolveClusterRequest(Cluster cluster) {
        return FetcherApiProto.TResolveClusterRequest.newBuilder()
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.CONDUCTOR_GROUP, cluster.getConductorGroups()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.CONDUCTOR_TAG, cluster.getConductorTags()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.NANNY, cluster.getNannyGroups()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.YP, cluster.getYpClusters()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.QLOUD, cluster.getQloudGroups()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.NETWORK, cluster.getNetworks()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.INSTANCE_GROUP, cluster.getInstanceGroups()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.HOSTS, cluster.getHosts()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.HOST_URL, cluster.getHostUrls()))
            .addGroups(toGroupConf(FetcherApiProto.TGroupConf.EGroupType.CLOUD_DNS, cluster.getCloudDns()))
            .build();
    }

    private static <T> FetcherApiProto.TGroupConf toGroupConf(
        FetcherApiProto.TGroupConf.EGroupType type,
        List<T> hosts)
    {
        final String jsonConfig;
        try {
            jsonConfig = jsonMapper.writeValueAsString(hosts);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("failed to convert to JSON group config: " + type);
        }

        return FetcherApiProto.TGroupConf.newBuilder()
            .setType(type)
            .setJsonConfig(jsonConfig)
            .build();
    }
}
