package ru.yandex.solomon.gateway.api.internal;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ProjectMenuConfig;
import ru.yandex.solomon.core.conf.ProjectMenuManager;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.gateway.api.internal.dto.WwwMenuDto;
import ru.yandex.solomon.gateway.api.internal.dto.WwwMenuItemDto;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.shard.ShardKey;

/**
 * Special controller to support menu in old UI
 *
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal/projects/{projectId}/menu", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ProjectMenuManager.class })
public class InternalMenuController {

    @Autowired
    private ProjectMenuManager menuManager;
    @Autowired
    private SolomonConfHolder solomonConfHolder;
    @Autowired
    private ClusterMapper clusterMapper;
    @Autowired
    private Authorizer authorizer;

    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<WwwMenuDto> getMenuMatchingLabels(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody Map<String, String> body)
    {
        String labels = body.getOrDefault("labels", "");
        Labels parsedLabels = LabelsFormat.parse(labels);

        String shardId = getShardIdFromLabels(parsedLabels);

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                .thenCompose(aVoid -> menuManager.getProjectMenu(projectId))
                .thenApply(projectMenu -> {
                    ProjectMenuConfig projectMenuConfig = new ProjectMenuConfig(projectMenu);
                    List<WwwMenuItemDto> menuItems = projectMenuConfig.constructMenu(parsedLabels)
                            .stream()
                            .map(WwwMenuItemDto::new)
                            .collect(Collectors.toList());
                    return new WwwMenuDto(menuItems, shardId, clusterMapper.knownClusterIds());
                });
    }

    private String getShardIdFromLabels(Labels parsedLabels) {
        try {
            ShardKey shardKey = ShardKey.get(parsedLabels);
            ShardConfDetailed shard = solomonConfHolder.getConfOrThrow().findShardOrNull(shardKey);
            return shard == null ? "" : shard.getId();
        } catch (Exception e) {
            return "";
        }
    }
}
