package ru.yandex.solomon.gateway.api.internal;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ShardsManager;
import ru.yandex.solomon.gateway.api.v2.dto.LabelValuesResponseDto;
import ru.yandex.solomon.gateway.api.v2.dto.ValidationUtils;
import ru.yandex.solomon.gateway.api.v2.managers.ShardLabelsManager;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ShardLabelsManager.class })
public class InternalShardsController {

    @Autowired
    private ShardLabelsManager shardLabelsManager;
    @Autowired
    private ShardsManager shardsManager;
    @Autowired
    private Authorizer authorizer;


    @RequestMapping(path = "/projects/{projectId}/shardLabels", method = RequestMethod.GET)
    CompletableFuture<LabelValuesResponseDto> findShardLabelValues(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "names", defaultValue = "") List<String> names,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "_validationFilter", defaultValue = "ALL") LabelValidationFilter validationFilter,
        @RequestParam(value = "limit", defaultValue = "1000") int limit)
    {
        ValidationUtils.validateLabelNames(names);

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenApply(aVoid -> {
                Selectors selectors = Selectors.parse(selectorsStr)
                    .toBuilder()
                    .addOverride(LabelKeys.PROJECT, projectId)
                    .build();

                var model = shardLabelsManager.findShardLabelValues(names, selectors, text, validationFilter, limit);
                return LabelValuesResponseDto.fromModel(model, selectors, validationFilter);
            });
    }

    /**
     * Force services (fetcher, coremon) to reload given shard.
     */
    @RequestMapping(path = "/projects/{projectId}/shards/{shardId}/reload", method = RequestMethod.POST)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> reloadShard(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("shardId") String shardId,
        @RequestParam(value = "host", required = false) String hostFilter,
        @RequestParam(value = "service", required = false) String serviceFilter)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE_INTERNALS)
            .thenCompose(aVoid -> shardsManager.reloadShard(projectId, "", shardId, hostFilter, serviceFilter));
    }
}
