package ru.yandex.solomon.gateway.api.internal;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.GraphsManager;
import ru.yandex.solomon.gateway.api.internal.dto.GraphOrDashboardItemDto;
import ru.yandex.solomon.labels.LabelsFormat;


/**
 * Special controller to support matching graphs by labels.
 *
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ GraphsManager.class })
public class MatchingGraphsController {

    @Autowired
    private GraphsManager graphsManager;
    @Autowired
    private Authorizer authorizer;

    @RequestMapping(path = "/api/internal/projects/{projectId}/matchingGraphs", method = RequestMethod.POST)
    CompletableFuture<List<GraphOrDashboardItemDto>> getGraphsMatchingLabels(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody Map<String, String> body)
    {
        String labels = body.getOrDefault("labels", "");
        Labels parsedLabels = LabelsFormat.parse(labels);
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
                .thenCompose(aVoid -> graphsManager.getAllGraphsMatchingLabels(projectId, "", parsedLabels))
                .thenApply(graphs -> graphs.stream()
                        .map(graph -> GraphOrDashboardItemDto.fromModel(graph, parsedLabels))
                        .collect(Collectors.toList()));
    }
}
