package ru.yandex.solomon.gateway.api.internal;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.gateway.backend.client.www.page.GraphHealth;
import ru.yandex.solomon.gateway.shardHealth.ShardHealthChecker;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.shard.ShardKey;

/**
 * Special controller to show matching shard info in old UI
 *
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal/projects/{projectId}/matchingShard", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ShardHealthChecker.class })
public class MatchingShardController {
    @Autowired
    private SolomonConfHolder solomonConfHolder;
    @Autowired
    private ShardHealthChecker shardHealthChecker;
    @Autowired
    private Authorizer authorizer;

    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<GraphHealth> getShardMatchingLabels(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "labels", defaultValue = "") String labels)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenApply(account -> {
                Labels parsedLabels = LabelsFormat.parse(labels);

                ShardConfDetailed shard = getShardFromLabels(parsedLabels);

                if (shard == null) {
                    return new GraphHealth("", "", 0);
                } else {
                    double shardHealthValue = shardHealthChecker.getShardHealthById(shard.getNumId());
                    return new GraphHealth(
                        shard.getProjectId(),
                        shard.getId(),
                        shardHealthValue);
                }
            });
    }

    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<GraphHealth> getShardMatchingLabelsPost(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody Map<String, String> body)
    {
        String labels = body.getOrDefault("labels", "");
        return getShardMatchingLabels(subject, projectId, labels);
    }

    @Nullable
    private ShardConfDetailed getShardFromLabels(Labels parsedLabels) {
        try {
            ShardKey shardKey = ShardKey.get(parsedLabels);
            return solomonConfHolder.getConfOrThrow().findShardOrNull(shardKey);
        } catch (Exception e) {
            return null;
        }
    }
}
