package ru.yandex.solomon.gateway.api.internal;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.cloud.session.SessionClient;
import ru.yandex.cloud.token.IamOauthClient;
import ru.yandex.solomon.auth.openid.OpenIdCookies;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.util.http.HttpUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * Read more about <a href="https://wiki.yandex-team.ru/cloud/iamrm/guides/oauthusage/">How to start using OAuth</a> and
 * <a href="https://wiki.yandex-team.ru/cloud/iamrm/services/API-autentifikacii-Oblaka/">Cloud authentication</a>.
 *
 * @author Sergey Polovko
 */
@RestController
@ConditionalOnBean(TGatewayCloudConfig.class)
public class OpenIdCallbackController {

    private final IamOauthClient oauthClient;
    private final SessionClient sessionClient;
    private final TAuthConfig.TOpenIdConfig openIdConfig;

    @Autowired
    public OpenIdCallbackController(IamOauthClient oauthClient, SessionClient sessionClient, TAuthConfig authConfig) {
        this.oauthClient = oauthClient;
        this.sessionClient = sessionClient;
        this.openIdConfig = authConfig.getOpenIdConfig();
    }

    @RequestMapping(path = "/oauth_cb", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<Void>> getOAuthTokenOld(
            @RequestParam("code") String code,
            @RequestParam("state") String state,
            ServerHttpRequest request)
    {
        String cookieHeader = request.getHeaders().getFirst("Cookie");
        String ycSession = HttpUtils.cookieValue(cookieHeader, OpenIdCookies.YC_SESSION);
        if (Strings.isNullOrEmpty(ycSession)) {
            // user does not have 'yc_session' cookie, so session definitely
            // mus be created
            return createSession(code, openIdConfig.getDomain());
        }

        String savedState = HttpUtils.cookieValue(cookieHeader, OpenIdCookies.YC_SESSION_STATE);
        if (Strings.isNullOrEmpty(savedState) || !savedState.equals(state)) {
            // wrong or empty 'yc_session_state' cookie, try to create new session
            return createSession(code, openIdConfig.getDomain());
        }

        // check session by provided 'yc_session' cookie
        return sessionClient.check(openIdConfig.getDomain(), cookieHeader, openIdConfig.getFederationId())
            .thenCompose(session -> {
                if (session.isPresent()) {
                    // if session is present simply redirect to main page
                    return completedFuture(redirectTo(getRetPath(), List.of()));
                }

                // else try to create new session
                return createSession(code, openIdConfig.getDomain());
            });
    }

    private CompletableFuture<ResponseEntity<Void>> createSession(String code, String domain) {
        return oauthClient.fetchAccessToken(code)
                .thenCompose(accessToken -> sessionClient.create(domain, accessToken))
                .thenApply(session -> {
                    if (session.isAbsent()) {
                        // something went wrong, and user must be redirected to
                        // OpenId provider again
                        return redirectTo(session.asAbsent().getRedirectUrl(), List.of());
                    }
                    return redirectTo(getRetPath(), session.asCreated().getCookies());
                });
    }

    private String getRetPath() {
        // TODO: retrieve return path from cookies
        return "https://" + openIdConfig.getDomain() + "/";
    }

    private static ResponseEntity<Void> redirectTo(String location, List<String> cookies) {
        var headers = new HttpHeaders();
        headers.add(HttpHeaders.LOCATION, location);
        for (String cookie : cookies) {
            headers.add(HttpHeaders.SET_COOKIE, cookie);
        }
        headers.add(HttpHeaders.SET_COOKIE, OpenIdCookies.removeSessionState());
        return ResponseEntity.status(HttpStatus.SEE_OTHER)
                .headers(headers)
                .build();
    }
}
