package ru.yandex.solomon.gateway.api.internal;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.core.conf.ViewHistoryManager;
import ru.yandex.solomon.core.db.model.ViewHistory;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.internal.dto.ViewHistoryDto;
import ru.yandex.solomon.gateway.api.internal.dto.ViewHistoryResultDto;
import ru.yandex.solomon.util.text.TextWithNumbersComparator;

import static ru.yandex.solomon.gateway.viewHistory.ViewHistoryUrlUtils.matchByUrl;
import static ru.yandex.solomon.gateway.viewHistory.ViewHistoryUrlUtils.matchByUrlPcsArgs;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ViewHistoryManager.class })
public class PinsController {

    @Autowired
    private ViewHistoryManager viewHistoryManager;

    @RequestMapping(path = "/pinsByUrl", method = RequestMethod.POST)
    CompletableFuture<List<ViewHistoryDto>> getPinsPost(
        @RequireAuth AuthSubject subject,
        @RequestBody Map<String, String> body)
    {
        String url = body.getOrDefault("url", "");
        return viewHistoryManager.findAll(subject.getUniqueId(), matchByUrlPcsArgs(url))
            .thenApply(result -> result.stream()
                .sorted(Comparator.comparing(ViewHistory::getPageTitle, TextWithNumbersComparator.instance))
                .map(ViewHistoryDto::fromModel)
                .collect(Collectors.toList()));
    }

    @RequestMapping(path = "/pinByUrl", method = RequestMethod.POST)
    CompletableFuture<ViewHistoryResultDto> getPinByUrl(
        @RequireAuth AuthSubject subject,
        @RequestBody Map<String, String> body)
    {
        String url = body.getOrDefault("url", "");
        String login = subject.getUniqueId();
        return findIdInHistoryByQueryArgs(url, login)
            .thenApply(ViewHistoryResultDto::fromModel);
    }

    private CompletableFuture<String> findIdInHistoryByQueryArgs(String url, String login) {
        return viewHistoryManager.findFirst(login, matchByUrl(url))
            .thenApply(history -> history.map(ViewHistory::getId).orElse(""));
    }

    @RequestMapping(path = "/pins", method = RequestMethod.POST)
    CompletableFuture<ViewHistoryDto> createPin(
        @RequireAuth AuthSubject subject,
        @RequestBody ViewHistoryDto dto)
    {
        ViewHistory model = dto.toModel(subject.getUniqueId());
        return viewHistoryManager.updateHistory(model, matchByUrl(model.getUrl()))
            .thenApply(updatedO -> {
                ViewHistory updated = updatedO.orElseThrow(() -> new BadRequestException(
                    String.format("failed to update history %s", model.toUserString())));
                return ViewHistoryDto.fromModel(updated);
            });
    }

    @RequestMapping(path = "/pins/{id}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deletePin(
        @RequireAuth AuthSubject subject,
        @PathVariable("id") String id)
    {
        return viewHistoryManager.deleteByLoginAndId(subject.getUniqueId(), id)
            .thenAccept(deleted -> {
                if (!deleted) {
                    String message = "failed to delete pin " + id + " for user " + subject.getUniqueId();
                    throw new RuntimeException(message);
                }
            });
    }
}
