package ru.yandex.solomon.gateway.api.internal;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.internal.dto.ProjectPermissionsDto;

/**
 * Special endpoint to resolve project and their permissions for Classic UI only.
 *
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal/projects/{projectId}/permissions", produces = MediaType.APPLICATION_JSON_VALUE)
public class ProjectPermissionsController {
    private final Authorizer authorizer;
    private final ProjectsDao projectsDao;
    private final boolean isIdm;

    @Autowired
    public ProjectPermissionsController(
            Authorizer authorizer,
            ProjectsDao projectsDao,
            Optional<TIdmConfig> idmConfig)
    {
        this.authorizer = authorizer;
        this.projectsDao = projectsDao;
        this.isIdm = idmConfig.isPresent();
    }

    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<ProjectPermissionsDto> getPermissions(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId) {
        if (subject.getAuthType() == AuthType.IAM ||
                subject.getAuthType() == AuthType.OpenId ||
                isIdm) {
            return resolveProjectPermissionsUsingAuthorizer(subject, projectId);
        }
        // It's necessary to save old behavior when permissions in UI were computed from non-cached project ACL.
        // When you want to implement IDM support or other authorization source, you may drop this behavior.
        return getProjectPermissionsUsingProjectAcl(subject, projectId);
    }

    private CompletableFuture<ProjectPermissionsDto> getProjectPermissionsUsingProjectAcl(
            AuthSubject subject,
            String projectId)
    {
        String login = AuthSubject.getLogin(subject)
                .orElseThrow(() -> new IllegalStateException("empty login in " + subject));

        return projectsDao.findById(projectId).thenApply(projectOpt -> {
            if (projectOpt.isEmpty()) {
                throw new NotFoundException("no found project with id " + projectId);
            }
            Project project = projectOpt.get();
            ProjectPermissionsDto dto = new ProjectPermissionsDto();
            dto.authorizationType = AuthorizationType.PROJECT_ACL;
            if ("junk".equals(project.getId()) || SolomonTeam.isMember(login) || login.equals(project.getOwner())) {
                dto.canRead = true;
                dto.canUpdate = true;
                dto.canDelete = true;
            } else {
                dto.canRead = !project.isOnlyAuthRead() || project.getAcl().getCanRead().contains(login);
                dto.canUpdate = project.getAcl().getCanUpdate().contains(login);
                dto.canDelete = project.getAcl().getCanDelete().contains(login);
            }
            return dto;
        });
    }

    private CompletableFuture<ProjectPermissionsDto> resolveProjectPermissionsUsingAuthorizer(AuthSubject subject, String projectId) {
        var canReadFuture = resolvePermission(subject, projectId, Permission.CONFIGS_GET);
        var canUpdateFuture = resolvePermission(subject, projectId, Permission.CONFIGS_UPDATE);
        var canDeleteFuture = resolvePermission(subject, projectId, Permission.CONFIGS_DELETE);
        var canRoleUpdate = isIdm
                ? resolvePermission(subject, projectId, Permission.ROLES_UPDATE)
                : CompletableFuture.completedFuture((Pair<Permission, AuthorizationType>) null);
        var futures = List.of(canReadFuture, canUpdateFuture, canDeleteFuture, canRoleUpdate);

        return CompletableFutures.allOf(futures).thenApply(responses -> {
            var permissions = responses.stream()
                    .filter(Objects::nonNull)
                    .map(Pair::getLeft)
                    .collect(Collectors.toSet());

            var authSource = responses.stream()
                    .filter(Objects::nonNull)
                    .map(Pair::getRight)
                    .findAny()
                    .orElse(AuthorizationType.UNKNOWN);

            var dto = new ProjectPermissionsDto();
            dto.authorizationType = authSource;
            dto.canRead = permissions.contains(Permission.CONFIGS_GET);
            dto.canUpdate = permissions.contains(Permission.CONFIGS_UPDATE);
            dto.canDelete = permissions.contains(Permission.CONFIGS_DELETE);
            dto.canUpdateRoles = permissions.contains(Permission.ROLES_UPDATE);
            return dto;
        });
    }

    private CompletableFuture<Pair<Permission, AuthorizationType>> resolvePermission(AuthSubject subject, String projectId, Permission permission) {
        return authorizer.authorize(subject, projectId, permission).handle((account, t) -> {
            if (t != null) {
                return null;
            }
            return Pair.of(permission, account.getAuthorizationType());
        });
    }
}
