package ru.yandex.solomon.gateway.api.internal;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ProjectSettingsManager;
import ru.yandex.solomon.gateway.api.internal.dto.ProjectSettingsDto;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal/projects/{projectId}/settings", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import(ProjectSettingsManager.class)
@ParametersAreNonnullByDefault
public class ProjectSettingsController {

    @Autowired
    private Authorizer authorizer;
    @Autowired
    private ProjectSettingsManager projectSettingsManager;

    @RequestMapping(method = RequestMethod.GET)
    public CompletableFuture<ProjectSettingsDto> getProjectSettings(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                .thenCompose(account -> projectSettingsManager.getProjectSettings(projectId))
                .thenApply(ProjectSettingsDto::fromModel);
    }

    @RequestMapping(method = RequestMethod.PUT)
    public CompletableFuture<ProjectSettingsDto> updateUserSettings(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestBody Map<String, String> settings)
    {
        ProjectSettingsValidator.validateSettings(settings);

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
                .thenCompose(account -> projectSettingsManager.updateProjectSettings(projectId, settings))
                .thenApply(ProjectSettingsDto::fromModel);
    }

    @RequestMapping(path = "/{key}", method = RequestMethod.GET)
    public CompletableFuture<String> getUserSettingByKey(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("key") String key)
    {
        ProjectSettingsValidator.validateKey(key);
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                .thenCompose(account -> projectSettingsManager.getProjectSettingByKey(projectId, key));
    }

    @RequestMapping(path = "/{key}", method = RequestMethod.PUT, consumes = MediaType.TEXT_PLAIN_VALUE)
    public CompletableFuture<String> setUserSettingByKey(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("key") String key,
            @RequestBody String value)
    {
        ProjectSettingsValidator.validateSetting(key, value);
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
                .thenCompose(account -> projectSettingsManager.updateProjectSettingByKey(projectId, key, value));
    }

    @RequestMapping(path = "/{key}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    public CompletableFuture<Void> deleteUserSettingByKey(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("key") String key)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
                .thenCompose(account -> projectSettingsManager.deleteProjectSettingByKey(projectId, key));
    }
}
