package ru.yandex.solomon.gateway.api.internal;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.RoleListRequest;
import ru.yandex.monitoring.api.v3.UpdateRoleListRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.gateway.api.internal.dto.RolesDto;
import ru.yandex.solomon.gateway.api.v3.intranet.ProjectRoleService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal", produces = MediaType.APPLICATION_JSON_VALUE)
public class RolesController {

    private final ProjectRoleService projectRoleService;

    public RolesController(Optional<ProjectRoleService> projectRoleService) {
        this.projectRoleService = projectRoleService.orElse(null);
    }

    @ApiOperation(value = "List project roles")
    @GetMapping(path = "/{projectId}/roles")
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        RoleListRequest listRequest = RoleListRequest.newBuilder()
                .setProjectId(projectId)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return projectRoleService.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @ApiOperation(value = "Update project roles. Can add and delete roles")
    @PostMapping(path = "/{projectId}/roles")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody byte[] body)
    {
        UpdateRoleListRequest.Builder builder = UpdateRoleListRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        return projectRoleService.update(builder.build(), subject).thenApply(empty -> null);
    }

    @ApiOperation(value = "Check project roles update status")
    @PostMapping(path = "/{projectId}/roles/status")
    CompletableFuture<String> status(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        UpdateRoleListRequest.Builder builder = UpdateRoleListRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        return projectRoleService.status(builder.build(), subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @ApiOperation(
        value = "get user login and roles",
        notes = "Will use received from browser to authenticate user and provide its " +
            "login and common roles. Designed to be used only by Solomon UI.",
        response = RolesDto.class
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
    })
    @RequestMapping(path = "/roles", method = RequestMethod.GET)
    RolesDto roles(@RequireAuth AuthSubject subject) {
        RoleSet roles = getRoles(subject);
        RolesDto dto = new RolesDto();
        // Special behavior to support login overriding from old /api/internal/auth endpoint
        // for AsUser authentication only
        if (subject.getAuthType() == AuthType.AsUser && "unknown".equals(subject.getUniqueId())) {
            dto.login = System.getProperty("user.name");
            dto.roles = RoleSet.SYSTEM_ALL.toEnumSet();
        } else {
            dto.login = AuthSubject.getLogin(subject)
                    .orElseThrow(() -> new IllegalStateException("empty login in " + subject));
            dto.roles = roles.toEnumSet();
        }
        return dto;
    }

    static RoleSet getRoles(AuthSubject subject) {
        return SolomonTeam.isMember(subject)
            ? RoleSet.SYSTEM_ALL
            : RoleSet.EMPTY;
    }
}
