package ru.yandex.solomon.gateway.api.internal;

import java.util.concurrent.CompletableFuture;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.v3.cloud.priv.CreateServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.DeleteServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.GetServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.ListServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.SyncServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.UpdateServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.ValidateServiceDashboardRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;
import ru.yandex.solomon.gateway.api.v3alpha.cloud.priv.PrivServiceDashboardService;
import ru.yandex.solomon.util.collection.Nullables;

@RestController
@RequestMapping(path = "/api/internal/serviceDashboards", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ServiceDashboardsController {
    private final PrivServiceDashboardService serviceDashboardService;

    @Autowired
    public ServiceDashboardsController(PrivServiceDashboardService privServiceDashboardService) {
        this.serviceDashboardService = privServiceDashboardService;
    }

    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "service") String service,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListServiceDashboardRequest listRequest = ListServiceDashboardRequest.newBuilder()
                .setService(service)
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return serviceDashboardService.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateServiceDashboardRequest.Builder builder = CreateServiceDashboardRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        CreateServiceDashboardRequest request = builder.build();

        return serviceDashboardService.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/{serviceDashboardId}", method = RequestMethod.GET)
    CompletableFuture<String> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceDashboardId") String serviceDashboardId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetServiceDashboardRequest getRequest = GetServiceDashboardRequest.newBuilder()
                .setServiceDashboardId(serviceDashboardId)
                .build();
        return serviceDashboardService.get(getRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/{serviceDashboardId}", method = RequestMethod.PUT, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceDashboardId") String serviceDashboardId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        UpdateServiceDashboardRequest.Builder builder = UpdateServiceDashboardRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        UpdateServiceDashboardRequest updateRequest = builder
                .setServiceDashboardId(serviceDashboardId)
                .build();
        return serviceDashboardService.update(updateRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/validate", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> validate(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        ValidateServiceDashboardRequest.Builder builder = ValidateServiceDashboardRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        ValidateServiceDashboardRequest validateRequest = builder.build();
        return serviceDashboardService.validate(validateRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/{serviceDashboardId}/sync/validate", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> validateSync(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceDashboardId") String serviceDashboardId,
            @RequestParam(value = "dashboardId") String dashboardId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        SyncServiceDashboardRequest request = SyncServiceDashboardRequest.newBuilder()
                .setServiceDashboardId(serviceDashboardId)
                .setDashboardId(dashboardId)
                .build();
        return serviceDashboardService.validateSync(request, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/{serviceDashboardId}/sync", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> sync(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceDashboardId") String serviceDashboardId,
            @RequestParam(value = "dashboardId") String dashboardId,
            @RequestParam(value = "sourceDashboardLink") String sourceDashboardLink,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        SyncServiceDashboardRequest request = SyncServiceDashboardRequest.newBuilder()
                .setServiceDashboardId(serviceDashboardId)
                .setDashboardId(dashboardId)
                .setSourceDashboardLink(Nullables.orEmpty(sourceDashboardLink))
                .build();
        return serviceDashboardService.sync(request, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/{serviceDashboardId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteService(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceDashboardId") String serviceDashboardId)
    {
        DeleteServiceDashboardRequest deleteRequest = DeleteServiceDashboardRequest.newBuilder()
                .setServiceDashboardId(serviceDashboardId)
                .build();

        return serviceDashboardService.delete(deleteRequest, subject).thenApply(empty -> null);
    }
}
