package ru.yandex.solomon.gateway.api.internal;

import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.TargetsStatusHelper;
import ru.yandex.solomon.fetcher.client.FetcherClient;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.ShardTargetStatusDto;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.net.KnownDc;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

/**
 * @author Ivan Zhukov
 */
@Api(tags = "shards")
@RestController
@RequestMapping(path = "/api/internal/serviceProviders/{serviceProviderId}", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ServiceProviderController {
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private FetcherClient fetcherClient;

    // used only for swagger documentation generation
    private static final class ServiceProviderTargetStatusDtoPage extends PagedResultDto<ShardTargetStatusDto> {}
    private static final Logger logger = LoggerFactory.getLogger(ServiceProviderController.class);

    @ApiOperation(
        value = "get service provider's targets statuses",
        notes = "This action will retrieve paged list of statuses of pulled service provider targets",
        response = ServiceProviderTargetStatusDtoPage.class
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "provider was not found"),
    })
    @RequestMapping(path = "/targets", method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<ShardTargetStatusDto>> targetsStatus(
        @RequireAuth AuthSubject subject,
        @PathVariable("serviceProviderId") String serviceProviderId,
        @RequestParam(name = "hostGlob", defaultValue = "") String hostGlob,
        @RequestParam(name = "dc", required = false) @Nullable KnownDc dc,
        @RequestParam(name = "status", required = false) @Nullable String statusStr,
        PageOptions pageOptions)
    {
        boolean notOkStatus = "NOT_OK".equals(statusStr);
        UrlStatusType status = (notOkStatus || StringUtils.isEmpty(statusStr))
            ? null
            : UrlStatusType.valueOf(statusStr);

        FetcherApiProto.TargetsStatusRequest.Builder requestBuilder = TargetsStatusHelper.requestBuilder(
            hostGlob, dc, status,
            notOkStatus, serviceProviderId);

        requestBuilder.setOffset(pageOptions.getOffset());
        if (pageOptions.isLimited()) {
            requestBuilder.setLimit(pageOptions.getSize());
        }

        final var request = requestBuilder.build();

        return authorizer.authorize(subject, serviceProviderId, Permission.CONFIGS_LIST)
                .thenCompose(aVoid -> {
                    var fs = new ArrayList<CompletableFuture<FetcherApiProto.TargetsStatusResponse>>();

                    for (var host: fetcherClient.getKnownHosts()) {
                        var f = fetcherClient.targetsStatus(host, request)
                                .exceptionally(e -> {
                                    logger.error("failed to get target status for service provider {} from {}. reason: {}",
                                            serviceProviderId, host, e);
                                    return FetcherApiProto.TargetsStatusResponse.getDefaultInstance();
                                });
                        fs.add(f);
                    }

                    return CompletableFutures.allOf(fs);
                })
                .thenApply(responses -> {
                    var targets = new ArrayList<FetcherApiProto.TargetStatus>();
                    int totalCount = 0;

                    for (var resp: responses) {
                        targets.addAll(resp.getTargetsList());
                        totalCount += resp.getTotalCount();
                    }

                    return FetcherApiProto.TargetsStatusResponse.newBuilder()
                            .addAllTargets(targets)
                            .setTotalCount(totalCount)
                            .build();
                })
                .thenApply(response -> PagedResult.of(response.getTargetsList(), pageOptions, response.getTotalCount()))
                .thenApply(res -> PagedResultDto.fromModel(res, ShardTargetStatusDto::fromModel));
    }
}
