package ru.yandex.solomon.gateway.api.internal;

import java.util.concurrent.CompletableFuture;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.v3.priv.CreateServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.DeleteServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.GetServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.ListServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.UpdateServiceProviderRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;
import ru.yandex.solomon.gateway.api.v3alpha.priv.ServiceProviderService;

@RestController
@RequestMapping(path = "/api/internal/serviceProviders", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ServiceProvidersController {
    private final ServiceProviderService serviceProviderService;

    @Autowired
    public ServiceProvidersController(ServiceProviderService serviceProviderService) {
        this.serviceProviderService = serviceProviderService;
    }


    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        return serviceProviderService.list(ListServiceProviderRequest.newBuilder()
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build(), subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateServiceProviderRequest.Builder builder = CreateServiceProviderRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        CreateServiceProviderRequest createRequest = builder.build();
        return serviceProviderService.create(createRequest, subject)
        .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/{serviceProviderId}", method = RequestMethod.GET)
    CompletableFuture<String> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceProviderId") String serviceProviderId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetServiceProviderRequest getRequest = GetServiceProviderRequest.newBuilder()
                .setServiceProviderId(serviceProviderId)
                .build();
        return serviceProviderService.get(getRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/{serviceProviderId}", method = RequestMethod.PUT)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceProviderId") String serviceProviderId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        UpdateServiceProviderRequest.Builder builder = UpdateServiceProviderRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setServiceProviderId(serviceProviderId);
        UpdateServiceProviderRequest updateRequest = builder.build();
        return serviceProviderService.update(updateRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/{serviceProviderId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteService(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceProviderId") String serviceProviderId)
    {
        DeleteServiceProviderRequest deleteRequest = DeleteServiceProviderRequest.newBuilder()
                .setServiceProviderId(serviceProviderId)
                .build();

        return serviceProviderService.delete(deleteRequest, subject).thenApply(empty -> null);
    }
}
