package ru.yandex.solomon.gateway.api.internal;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.internal.dto.StockpileDataResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.ValidationUtils;
import ru.yandex.solomon.gateway.data.StockpileMetricClient;
import ru.yandex.solomon.gateway.data.StockpileMetricDataRequest;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.util.time.Interval;


/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal/projects/{projectId}/stockpile/sensor", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class StockpileMetricController {

    @Autowired
    private StockpileMetricClient stockpileMetricClient;
    @Autowired
    private ClusterMapper clusterMapper;
    @Autowired
    private InternalAuthorizer authorizer;

    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<StockpileDataResultDto> readMetric(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "name", defaultValue = "") String metricName,
        @RequestParam("labels") String labelsParam,
        @RequestParam("from") Instant from,
        @RequestParam("to") Instant to,
        @RequestParam(value = "forceCluster", defaultValue = "") String forceCluster)
    {
        Instant deadline = Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);
        Labels labels = LabelsFormat.parse(labelsParam);

        validateLabels(labels);
        ValidationUtils.validateInterval(from, to);

        return authorizer.authorize(subject)
            .thenCompose(aVoid -> {
                StockpileMetricDataRequest request = new StockpileMetricDataRequest(
                    metricName,
                    labels,
                    Interval.millis(from.toEpochMilli(), to.toEpochMilli()),
                    forceCluster);

                return stockpileMetricClient.read(projectId, request, deadline, AuthSubject.getLogin(subject, subject.getUniqueId()))
                    .thenApply(response -> StockpileDataResultDto.fromModel(response));
            });
    }

    private static void validateLabels(Labels labels) {
        if (labels.isEmpty()) {
            throw new BadRequestException("empty labels");
        }

        if (!labels.hasKey(LabelKeys.CLUSTER) || !labels.hasKey(LabelKeys.SERVICE)) {
            throw new BadRequestException("labels doesn't contain shard labels: " + labels);
        }
    }
}
