package ru.yandex.solomon.gateway.api.internal;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.core.conf.UserSettingsManager;
import ru.yandex.solomon.gateway.api.internal.dto.UserSettingsDto;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/api/internal/settings", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import(UserSettingsManager.class)
@ParametersAreNonnullByDefault
public class UserSettingsController {

    @Autowired
    private UserSettingsManager userSettingsManager;

    @RequestMapping(method = RequestMethod.GET)
    public CompletableFuture<UserSettingsDto> getUserSettings(@RequireAuth AuthSubject subject) {
        return userSettingsManager.getUserSettings(subject.getUniqueId())
            .thenApply(UserSettingsDto::fromModel);
    }

    @RequestMapping(method = RequestMethod.PUT)
    public CompletableFuture<UserSettingsDto> updateUserSettings(
        @RequireAuth AuthSubject subject,
        @RequestBody Map<String, String> settings)
    {
        UserSettingsValidator.validateSettings(settings);

        return userSettingsManager.updateUserSettings(subject.getUniqueId(), settings)
            .thenApply(UserSettingsDto::fromModel);
    }

    @RequestMapping(path = "/{key}", method = RequestMethod.GET)
    public CompletableFuture<String> getUserSettingByKey(
        @RequireAuth AuthSubject subject,
        @PathVariable("key") String key)
    {
        UserSettingsValidator.validateKey(key);
        return userSettingsManager.getUserSettingByKey(subject.getUniqueId(), key);
    }

    @RequestMapping(path = "/{key}", method = RequestMethod.PUT, consumes = MediaType.TEXT_PLAIN_VALUE)
    public CompletableFuture<String> setUserSettingByKey(
        @RequireAuth AuthSubject subject,
        @PathVariable("key") String key,
        @RequestBody String value)
    {
        UserSettingsValidator.validateSetting(key, value);

        return userSettingsManager.updateUserSettingByKey(subject.getUniqueId(), key, value);
    }

    @RequestMapping(path = "/{key}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    public CompletableFuture<Void> deleteUserSettingByKey(
        @RequireAuth AuthSubject subject,
        @PathVariable("key") String key)
    {
        return userSettingsManager.deleteUserSettingByKey(subject.getUniqueId(), key);
    }
}
