package ru.yandex.solomon.gateway.api.internal.yasm;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import io.grpc.Status;

import ru.yandex.ambry.dto.TagsField;
import ru.yandex.monitoring.gateway.DataQuery;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.gateway.data.DownsamplingOptions;
import ru.yandex.solomon.gateway.data.DownsamplingType;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.util.time.Interval;
import ru.yandex.solomon.yasm.expression.ast.YasmAst;
import ru.yandex.solomon.yasm.expression.grammar.YasmExpression;
import ru.yandex.solomon.yasm.expression.grammar.YasmSelRenderer;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class RequestConverter {
    private static final Pattern HOST_AGGREGATES = Pattern.compile("^[A-Z0-9.]+$");

    @VisibleForTesting
    public static CompletableFuture<DataRequest> makeDataClientRequest(DataQuery dataQuery) {
        return makeDataClientRequest(dataQuery, "yasm_", "");
    }

    public static CompletableFuture<DataRequest> makeDataClientRequest(DataQuery dataQuery, String yasmProjectPrefix, String subjectId) {
        StringBuilder program = new StringBuilder();

        HashMap<String, List<String>> tags = new HashMap<>(TagsField.parseJoinedTags(dataQuery.getTags()));

        List<String> itypes = tags.getOrDefault("itype", List.of());
        if (itypes.size() != 1) {
            return failedFuture(Status.INVALID_ARGUMENT
                    .withDescription("Expected exactly one itype, got: " + itypes)
                    .asRuntimeException());
        }

        if (dataQuery.getHosts().isEmpty()) {
            return failedFuture(Status.INVALID_ARGUMENT
                    .withDescription("Hosts field is missing from request")
                    .asRuntimeException());
        }

        String host = dataQuery.getHosts();
        boolean wasAggregated = isGroupOrMetagroup(host);
        if (wasAggregated) {
            tags.put("group", List.of(host));
        } else {
            tags.put("host", List.of(host));
        }
        program.append(YasmSelRenderer.renderTags(tags, yasmProjectPrefix));
        program.append('\n');

        YasmAst ast = YasmExpression.parse(dataQuery.getExpression());
        YasmSelRenderer.RenderResult renderResult = YasmSelRenderer.render(ast, yasmProjectPrefix);

        for (var entry : renderResult.constants.entrySet()) {
            String key = entry.getKey();
            SelValue value = entry.getValue();
            program.append("let ").append(key).append(" = ").append(value).append(";\n");
        }
        program.append("return ").append(renderResult.expression).append(';');

        var requestBuilder = DataRequest.newBuilder()
                .setVersion(SelVersion.HISTOGRAM_FUNCTIONS_DONT_MERGE_3)
                .setProjectId(yasmProjectPrefix + itypes.get(0))
                .setProgram(program.toString())
                .setDeadline(Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS))
                .setProducer(RequestProducer.USER)
                .setInterval(Interval.millis(dataQuery.getFromMillis(), dataQuery.getToMillis()))
                .setSubjectId(subjectId);

        if (dataQuery.getGridMillis() != 0L) {
            requestBuilder.setDownsampling(DownsamplingOptions.newBuilder()
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE) // TODO: until SOLOMON-5914
                    .setDownsamplingAggr(Aggregation.DEFAULT_AGGREGATION)
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setGridMillis(dataQuery.getGridMillis())
                    .build());
        }

        return completedFuture(requestBuilder.build());
    }

    static boolean isGroupOrMetagroup(String host) {
        return HOST_AGGREGATES.matcher(host).matches();
    }
}
