package ru.yandex.solomon.gateway.api.internal.yasm;

import java.util.function.ToDoubleFunction;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.doubles.DoubleArrayList;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.gateway.AlertStatusResponse;
import ru.yandex.monitoring.gateway.DataQuery;
import ru.yandex.monitoring.gateway.Timeseries;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.gateway.data.DataResponse;
import ru.yandex.solomon.gateway.data.RandomSeriesGenerator;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.EmptyAggrGraphDataIterator;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;

import static ru.yandex.solomon.model.point.column.StockpileColumns.minColumnSet;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ResponseConverter {
    private static final Logger logger = LoggerFactory.getLogger(ResponseConverter.class);

    private static String throwableToString(Throwable t) {
        return t.getClass().getSimpleName() + ": " + t.getMessage();
    }

    static Timeseries exceptionToTimeseries(Throwable throwable) {
        Throwable unwrapped = CompletableFutures.unwrapCompletionException(throwable);
        return Timeseries.newBuilder()
                .addErrors(throwableToString(unwrapped))
                .build();
    }

    static Timeseries dataResponseToTimeseries(DataResponse dataResponse, DataQuery query) {
        SelValue result = dataResponse.getEvalResult();
        if (result.type().isGraphData()) {
            return graphDataToTimeseries(result.castToGraphData().getNamedGraphData());
        }
        if (result.type().isGraphDataVector()) {
            SelValueVector vector = result.castToVector();
            if (vector.valueArray().length == 0) {
                return Timeseries.getDefaultInstance();
            }
            if (vector.valueArray().length > 1) {
                return Timeseries.newBuilder()
                        .addErrors("Expression returned too many timeseries: got vector of size " + vector.valueArray().length)
                        .build();
            }
            return graphDataToTimeseries(result.castToVector().valueArray()[0].castToGraphData().getNamedGraphData());
        }
        return Timeseries.newBuilder()
                .addErrors("Expression returned unrepresentable value of type: " + result.type())
                .build();
    }

    @Nullable
    private static AggrGraphDataListIterator extractDgauge(NamedGraphData namedGraphData) {
        var points = namedGraphData.getAggrGraphDataArrayList();
        if (points.isEmpty()) {
            return EmptyAggrGraphDataIterator.INSTANCE;
        }

        if (namedGraphData.getDataType() == MetricType.DSUMMARY) {

        }

        if (MetricTypeTransfers.isAvailableTransfer(namedGraphData.getDataType(), MetricType.DGAUGE)) {
            return MetricTypeTransfers.of(namedGraphData.getDataType(), MetricType.DGAUGE, points.iterator());
        }

        return null;
    }

    private static AggrGraphDataListIterator extractFromSummary(AggrGraphDataListIterator iterator, ToDoubleFunction<SummaryDoubleSnapshot> extractor) {
        return new AggrGraphDataListIterator(StockpileColumns.minColumnSet(MetricType.DGAUGE)) {
            @Override
            public boolean next(AggrPoint point) {
                if (!iterator.next(point)) {
                    return false;
                }

                point.setValue(extractor.applyAsDouble(point.summaryDouble));
                point.clearField(StockpileColumn.DSUMMARY);

                return true;
            }
        };
    }

    private static Timeseries graphDataToTimeseries(NamedGraphData namedGraphData) {
        var points = namedGraphData.getAggrGraphDataArrayList();
        if (points.isEmpty()) {
            return Timeseries.getDefaultInstance();
        }

        var iterator = extractDgauge(namedGraphData);

        if (iterator == null) {
            return Timeseries.newBuilder()
                    .addErrors("Cannot represent points of type: " + namedGraphData.getDataType())
                    .build();
        }

        var timeseries = Timeseries.newBuilder();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();

        LongArrayList timestamps = new LongArrayList(points.length());
        DoubleArrayList values = new DoubleArrayList(points.length());
        try {
            point.reset(minColumnSet(MetricType.DGAUGE));
            while (iterator.next(point)) {
                timestamps.add(point.getTsMillis());
                values.add(point.getValueDivided());
            }
        } finally {
            point.recycle();
        }

        timeseries.addAllTimestampsMillis(timestamps);
        timeseries.addAllValues(values);

        return timeseries.build();
    }

    static Timeseries addStubDataIfEmpty(Timeseries timeseries, DataQuery query) {
        if (timeseries.getTimestampsMillisCount() > 0 || timeseries.getErrorsCount() == 0) {
            return timeseries;
        }

        String key = query.getHosts() + ":" + query.getTags() + ":" + query.getExpression();
        long gridMillis = query.getGridMillis() != 0 ? query.getGridMillis() : YasmGatewayServiceImpl.DEFAULT_GRID_MILLIS;
        var iterator = new RandomSeriesGenerator(key, query.getFromMillis(), query.getToMillis(), gridMillis);

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();

        LongArrayList timestamps = new LongArrayList(iterator.estimatePointsCount());
        DoubleArrayList values = new DoubleArrayList(iterator.estimatePointsCount());
        try {
            point.reset(minColumnSet(MetricType.DGAUGE));
            while (iterator.next(point)) {
                timestamps.add(point.getTsMillis());
                values.add(point.getValueDivided());
            }
        } finally {
            point.recycle();
        }

        return timeseries.toBuilder()
                .addAllTimestampsMillis(timestamps)
                .addAllValues(values)
                .build();
    }

    static AlertStatusResponse.AlertStatus exceptionToAlertStatus(Throwable t) {
        Throwable unwrapped = CompletableFutures.unwrapCompletionException(t);
        return AlertStatusResponse.AlertStatus.newBuilder()
                .setDescription(throwableToString(unwrapped))
                .build();
    }
}
