package ru.yandex.solomon.gateway.api.internal.yasm;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.primitives.Doubles;
import io.grpc.Status;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.gateway.AlertStatusRequest;
import ru.yandex.monitoring.gateway.AlertStatusResponse;
import ru.yandex.monitoring.gateway.QueryTopRequest;
import ru.yandex.monitoring.gateway.QueryTopResponse;
import ru.yandex.monitoring.gateway.ReadDataRequest;
import ru.yandex.monitoring.gateway.ReadDataResponse;
import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.config.protobuf.frontend.TYasmGatewayConfig;
import ru.yandex.solomon.gateway.data.DataClient;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.yasm.alert.converter.YasmAlertConverter.alertIdFromYasmAlertName;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class YasmGatewayServiceImpl implements YasmGatewayService {

    static final long DEFAULT_GRID_MILLIS = TimeUnit.SECONDS.toMillis(10);

    private final DataClient dataClient;
    private final AlertingClient alertingClient;
    private final String yasmProjectPrefix;

    public YasmGatewayServiceImpl(DataClient dataClient, AlertingClient alertingClient, TYasmGatewayConfig config) {
        this.dataClient = dataClient;
        this.alertingClient = alertingClient;
        this.yasmProjectPrefix = config.getYasmProjectsPrefix().isEmpty() ? "yasm_" : config.getYasmProjectsPrefix();
    }

    @Override
    public CompletableFuture<ReadDataResponse> readData(ReadDataRequest request, AuthSubject subject) {
        return request.getQueriesList().stream()
                .map(query -> RequestConverter.makeDataClientRequest(query, yasmProjectPrefix, AuthSubject.getLogin(subject, subject.getUniqueId()))
                        .thenCompose(dataClient::readData)
                        .thenApply(dataResponse -> ResponseConverter.dataResponseToTimeseries(dataResponse, query))
                        .exceptionally(ResponseConverter::exceptionToTimeseries)
                        .thenApply(timeseries -> ResponseConverter.addStubDataIfEmpty(timeseries, query))
                )
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(timeseriesList -> ReadDataResponse.newBuilder()
                        .addAllTimeseries(timeseriesList)
                        .build());
    }

    @Override
    public CompletableFuture<QueryTopResponse> queryTop(QueryTopRequest request) {
        return failedFuture(Status.UNIMPLEMENTED.asRuntimeException());
    }

    private static double parseDoubleOrNaN(@Nullable String s) {
        if (s != null) {
            Double val = Doubles.tryParse(s);
            if (val != null) {
                return val;
            }
        }
        return Double.NaN;
    }

    private static AlertStatusResponse.AlertStatus stateToAlertStatus(TReadEvaluationStateResponse response) {
        if (response.getRequestStatus() != ERequestStatusCode.OK) {
            return AlertStatusResponse.AlertStatus.newBuilder()
                    .setDescription(response.getRequestStatus() + ": " + response.getStatusMessage())
                    .build();
        }

        var builder = AlertStatusResponse.AlertStatus.newBuilder();
        var state = response.getState();
        switch (state.getStatus().getCode()) {
            case OK -> builder.setAlertState(AlertStatusResponse.AlertStatus.AlertState.OK);
            case WARN -> builder.setAlertState(AlertStatusResponse.AlertStatus.AlertState.WARN);
            case ALARM -> builder.setAlertState(AlertStatusResponse.AlertStatus.AlertState.ALARM);
            case NO_DATA -> builder.setAlertState(AlertStatusResponse.AlertStatus.AlertState.NO_DATA);
            case ERROR -> builder.setAlertState(AlertStatusResponse.AlertStatus.AlertState.ERROR);
            default -> builder.setAlertState(AlertStatusResponse.AlertStatus.AlertState.UNDEFINED);
        }

        Map<String, String> annotations = state.getStatus().getAnnotationsMap();

        builder.setVisibleValue(parseDoubleOrNaN(annotations.getOrDefault("visible_value", "")));
        builder.setThresholdsValue(parseDoubleOrNaN(annotations.getOrDefault("thresholds_value", "")));

        builder.setDescription(annotations.getOrDefault("description", ""));

        return builder.build();
    }

    private CompletableFuture<TReadEvaluationStateRequest> makeReadAlertStateRequest(AlertStatusRequest.AlertKey alertKey) {
        if (alertKey.getItype().isEmpty()) {
            return failedFuture(Status.INVALID_ARGUMENT
                    .withDescription("Empty itype for alert: " + alertKey.getName())
                    .asRuntimeException());
        }
        return completedFuture(TReadEvaluationStateRequest.newBuilder()
                .setProjectId(yasmProjectPrefix + alertKey.getItype())
                .setAlertId(alertIdFromYasmAlertName(alertKey.getName()))
                .build());
    }

    @Override
    public CompletableFuture<AlertStatusResponse> alertStatus(AlertStatusRequest request) {
        return request.getAlertKeysList().stream()
                .map(alertKey -> makeReadAlertStateRequest(alertKey)
                        .thenCompose(alertingClient::readEvaluationState)
                        .thenApply(YasmGatewayServiceImpl::stateToAlertStatus)
                        .exceptionally(ResponseConverter::exceptionToAlertStatus))
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(alertStatuses -> AlertStatusResponse.newBuilder()
                        .addAllAlertStatuses(alertStatuses)
                        .build());
    }

}
