package ru.yandex.solomon.gateway.api.old.data;

import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.OptionalAuth;
import ru.yandex.solomon.auth.http.OptionalAuthMetrics;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.backend.client.data.WwwDataApiGet;
import ru.yandex.solomon.gateway.backend.client.data.WwwDataApiMetric;
import ru.yandex.solomon.gateway.backend.client.www.page.DownSamplingParams;
import ru.yandex.solomon.gateway.utils.Grids;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.www.BackendManager;
import ru.yandex.solomon.gateway.www.ser.WwwIntervalSerializer;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.math.doubles.AggregateFunctionType;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.util.time.Interval;
import ru.yandex.solomon.util.www.ser.WwwDurationSerializer;

/**
 * @author Stepan Koltsov
 *
 * @url https://wiki.yandex-team.ru/solomon/api/data
 */
@Controller
@RequestMapping("/data-api")
@Import({ BackendManager.class })
public class OldDataApiController {

    private static final int DEFAULT_POINT_COUNT = 500;

    @Autowired
    private Authorizer authorizer;
    @Autowired
    private BackendManager backendManager;
    @Autowired
    private ClusterMapper clusterMapper;

    @RequestMapping("/get")
    @ResponseBody
    public CompletableFuture<WwwDataApiGet> get(
        @OptionalAuth AuthSubject authSubject,
        @RequestParam(value = UserLinksBasic.B_QA, defaultValue = "") String b,
        @RequestParam(value = UserLinksBasic.E_QA, defaultValue = "") String e,
        @RequestParam(value = UserLinksBasic.NEED_POINTS_PARAM, defaultValue = "") String pointsParam,
        @RequestParam(value = UserLinksBasic.GRID, defaultValue = "") String gridParam,
        @RequestParam(value = UserLinksBasic.DOWNSAMPLING_AGGR, defaultValue = "") String downsamplingAggrParam,
        @RequestParam(value = UserLinksBasic.WHO) String who,
        @RequestParam(value = "forceDc", defaultValue = "") String forceDcParam,
        @RequestParam Map<String, String> params)
    {
        Instant now = Instant.now();

        OptionalAuthMetrics.INSTANCE.register("/data-api/get", authSubject);
        OldDataApiMetrics.INSTANCE.register("/data-api/get", authSubject, who);

        ShardKey shardKey = getShardKeyOrThrow(params);

        return authorizer.authorize(authSubject, shardKey.getProject(), Permission.DATA_READ)
            .thenCompose(account -> getImpl(now, b, e, shardKey, pointsParam, gridParam, downsamplingAggrParam, forceDcParam, params, getSubjectId(authSubject)));
    }

    private String getSubjectId(AuthSubject authSubject) {
        if (authSubject == AnonymousAuthSubject.INSTANCE) {
            return "OldDataApi_AnonymousAuthSubject";
        }
        return AuthSubject.getLogin(authSubject, authSubject.getUniqueId());
    }

    private CompletableFuture<WwwDataApiGet> getImpl(
        Instant now,
        String b,
        String e,
        ShardKey shardKey,
        String pointsParam,
        String gridParam,
        String downsamplingAggrParam,
        String forceDcParam,
        Map<String, String> params,
        String subjectId)
    {
        Interval interval = WwwIntervalSerializer.parseInterval(b, e).toInterval(now);

        DownSamplingParams downsamplingParams =
            computeDownsamplingParams(interval, pointsParam, gridParam, downsamplingAggrParam);

        String clusterId = clusterMapper.byParamOrNull(forceDcParam);

        return backendManager.processGet(
            shardKey, interval, downsamplingParams, clusterId, params, subjectId)
            .thenApply(response -> filterDataResponsePoints(response, pointsParam));
    }

    // Stockpile downsampling always returns two points,
    // so we just show last point in case of points=1
    private WwwDataApiGet filterDataResponsePoints(WwwDataApiGet response, String pointsParam) {
        if ("1".equals(pointsParam)) {
            List<WwwDataApiMetric> metrics = response.metrics.stream()
                .map(metric -> {
                    if (metric.getValues().size() > 1) {
                        var lastValue = metric.getValues().get(metric.getValues().size() - 1);
                        return new WwwDataApiMetric(
                            Labels.of(metric.getLabels()),
                            metric.getCreated(),
                            metric.isDeriv(),
                            Collections.singletonList(lastValue)
                        );
                    }

                    return metric;
                })
                .collect(Collectors.toList());

            return new WwwDataApiGet(metrics);
        }

        return response;
    }

    private DownSamplingParams computeDownsamplingParams(
        Interval interval,
        String pointsParam,
        String gridParam,
        String downsamplingAggrParam)
    {
        final AggregateFunctionType aggr;
        final long gridMillis;

        if (UserLinksBasic.DOWNSAMPLING_PARAM_RAW.equals(downsamplingAggrParam)) {
            aggr = AggregateFunctionType.AVG;
            gridMillis = 0;
        } else {
            aggr = AggregateFunctionType.byName(downsamplingAggrParam).orElse(AggregateFunctionType.AVG);
            gridMillis = computeGridMillis(interval, pointsParam, gridParam);
        }

        return new DownSamplingParams(gridMillis, aggr, OperationDownsampling.FillOption.NULL, false);
    }

    private static long computeGridMillis(
        Interval interval,
        String pointsParam,
        String gridParam)
    {
        long gridMillis;

        if (!StringUtils.isBlank(gridParam)) {
            Optional<Duration> gridOpt = WwwDurationSerializer.parseDuration(gridParam);
            if (!gridOpt.isPresent()) {
                throw new BadRequestException("failed to parse grid: " + gridParam);
            }
            gridMillis = gridOpt.get().toMillis();
        } else {
            final int points;

            if (!StringUtils.isBlank(pointsParam)) {
                try {
                    points = Integer.parseInt(pointsParam);
                } catch (Exception e) {
                    throw new BadRequestException("failed to parse points: " + pointsParam);
                }
            } else {
                points = DEFAULT_POINT_COUNT;
            }

            if (points < 0) {
                throw new BadRequestException("points must be positive number");
            }

            if (points == 0) {
                gridMillis = 0;
            } else if (points == 1) {
                gridMillis = interval.length().toMillis() * 2;
            } else {
                gridMillis = interval.length().toMillis() / (points - 1);
            }
        }

        if (gridMillis != 0) {
            gridMillis = Grids.roundMillisUpToGrid(gridMillis);
        }

        return gridMillis;
    }

    @RequestMapping("/sensors")
    @ResponseBody
    public CompletableFuture<WwwDataApiGet> metrics(
        @OptionalAuth AuthSubject authSubject,
        @RequestParam(value = UserLinksBasic.WHO) String who,
        @RequestParam Map<String, String> params)
    {
        OptionalAuthMetrics.INSTANCE.register("/data-api/sensors", authSubject);
        OldDataApiMetrics.INSTANCE.register("/data-api/sensors", authSubject, who);

        ShardKey shardKey = getShardKeyOrThrow(params);

        return authorizer.authorize(authSubject, shardKey.getProject(), Permission.METRICS_GET)
            .thenCompose(account -> metricsImpl(shardKey, params));
    }

    private CompletableFuture<WwwDataApiGet> metricsImpl(ShardKey shardKey, Map<String, String> params) {
        return backendManager.processMetrics(shardKey, params)
            .thenApply(WwwDataApiGet::new);
    }

    private static ShardKey getShardKeyOrThrow(Map<String, String> params) {
        String project = getParamAndRemoveFromMapOrThrow(LabelKeys.PROJECT, params);
        String cluster = getParamAndRemoveFromMapOrThrow(LabelKeys.CLUSTER, params);
        String service = getParamAndRemoveFromMapOrThrow(LabelKeys.SERVICE, params);
        return new ShardKey(project, cluster, service);
    }

    private static String getParamAndRemoveFromMapOrThrow(String name, Map<String, String> params) {
        for (String prefix : new String[]{UserLinksBasic.LABEL_NAME_QA_PREFIX, ""}) {
            String key = prefix + name;
            String value = params.remove(key);
            if (value != null) {
                return value;
            }
        }

        throw new BadRequestException("parameter '" + name + "' is required");
    }
}
