package ru.yandex.solomon.gateway.api.old.grafana;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.analytics.GraphDataLoadRequest;
import ru.yandex.solomon.expression.analytics.GraphDataLoader;
import ru.yandex.solomon.expression.analytics.PreparedProgram;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.gateway.backend.meta.search.MetaSearch;
import ru.yandex.solomon.gateway.backend.meta.search.PageMetricStatsUtils;
import ru.yandex.solomon.gateway.backend.storage.GraphDataClient;
import ru.yandex.solomon.gateway.backend.storage.MetricStorageRequest;
import ru.yandex.solomon.gateway.backend.storage.ReadMetrics;
import ru.yandex.solomon.gateway.backend.www.page.WwwLabelUtils;
import ru.yandex.solomon.labels.FormatLabel;
import ru.yandex.solomon.labels.LabelValues;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.math.doubles.AggregateFunctionType;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.TimeSeriesSizeCounter;
import ru.yandex.solomon.metrics.client.combined.DataLimits;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.util.text.TextComparator;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.solomon.util.time.Interval;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;


/**
 * @author Vladimir Gordiychuk
 */
@Component
@Import({
    SolomonConfHolder.class,
    MetaSearch.class,
    ReadMetrics.class,
    GrafanaMetrics.class,
})
public class GrafanaDataManager {
    private final Authorizer authorizer;
    private final SolomonConfHolder confHolder;
    private final MetaSearch metaSearch;
    private final GraphDataClient graphDataClient;
    private final ReadMetrics.ClientMetrics clientMetrics;
    private final GrafanaMetrics grafanaMetrics;
    private final FeatureFlagsHolder featureFlagsHolder;

    @Autowired
    public GrafanaDataManager(
        Authorizer authorizer,
        SolomonConfHolder confHolder,
        MetaSearch metaSearch,
        ReadMetrics clientMetrics,
        GrafanaMetrics grafanaMetrics,
        GraphDataClient graphDataClient,
        FeatureFlagsHolder featureFlagsHolder)
    {
        this.authorizer = authorizer;
        this.confHolder = confHolder;
        this.metaSearch = metaSearch;
        this.graphDataClient = graphDataClient;
        this.clientMetrics = clientMetrics.getClientMetrics("grafana");
        this.grafanaMetrics = grafanaMetrics;
        this.featureFlagsHolder = featureFlagsHolder;
    }

    public CompletableFuture<List<WwwTimeSeries>> search(AuthSubject authSubject, SearchRequest request) {
        Deadline deadline = Deadline.fromThreadLocal("grafanaExecExpr");

        GrafanaMetrics.DashboardMetrics metrics =
            grafanaMetrics.getDashboardMetrics(request.getDashboardIdOrAbsent());

        List<ParsedTarget> targets = request.getTargets()
            .stream()
            .filter(target -> StringUtils.isNotEmpty(target.getExpression()))
            .map(target -> new ParsedTarget(target.getRefId(), target.getExpression()))
            .collect(toList());

        CompletableFuture<List<WwwTimeSeries>> results = CompletableFutures.allOf(
            targets
                .stream()
                .map(target -> execExpr(authSubject, target, request.getInterval(), request.getGlobalGridMillis(), deadline, metrics))
                .collect(toList())
        ).thenApply(exprBatch -> exprBatch.stream().flatMap(Collection::stream).collect(toList()));

        metrics.asyncMetrics.forFuture(results);

        return results;
    }

    private CompletableFuture<List<WwwTimeSeries>> execExpr(
        AuthSubject authSubject,
        ParsedTarget target,
        Interval interval,
        long globalGridMillis,
        Deadline deadline,
        GrafanaMetrics.DashboardMetrics dashboardMetrics)
    {
        PreparedProgram preparedProgram = prepareProgram(target, interval);
        Optional<ShardKey> shardKeyOpt = preparedProgram.getLoadRequests()
            .stream()
            .map(request -> ShardSelectors.getShardKeyOrNull(request.getSelectors()))
            .filter(Objects::nonNull)
            .findFirst();

        if (shardKeyOpt.isEmpty()) {
            return CompletableFuture.completedFuture(Collections.emptyList());
        }

        ShardKey shardKey = shardKeyOpt.get();

        return authorizer.authorize(authSubject, shardKey.getProject(), Permission.DATA_READ).thenCompose(account -> {
            ShardConfDetailed shardConf = confHolder.getConfOrThrow().findShardByKey(shardKey);

            String shardId = shardConf.getId();
            var patchedRequests = roundDownsamplingToGrid(shardConf, globalGridMillis, preparedProgram.getLoadRequests());

            return resolveMetrics(patchedRequests.values(), deadline, dashboardMetrics)
                .thenCompose(metrics -> loadMetrics(shardId, metrics, deadline, getSubjectId(authSubject)))
                .thenApply(loader -> {
                    var patchedLoader = new PatchedGraphDataLoader(loader, patchedRequests);
                    Map<String, SelValue> evalResult = preparedProgram.evaluate(patchedLoader, Collections.emptyMap());
                    SelValue selValue = evalResult.get(preparedProgram.expressionToVarName(target.expression));

                    return WwwTimeSeriesConverter.toTimeSeries(selValue, target.salmonId);
                });
        });
    }

    private String getSubjectId(AuthSubject authSubject) {
        if (authSubject == AnonymousAuthSubject.INSTANCE) {
            return "Grafana_AnonymousAuthSubject";
        }
        return AuthSubject.getLogin(authSubject, authSubject.getUniqueId());
    }

    private Map<GraphDataLoadRequest, GraphDataLoadRequest> roundDownsamplingToGrid(ShardConfDetailed shardConf, long globalGridMillis, Collection<GraphDataLoadRequest> requests) {
        var result = new HashMap<GraphDataLoadRequest, GraphDataLoadRequest>();
        int gridSec = shardConf.getGridSec();
        if (!featureFlagsHolder.hasFlag(FeatureFlag.GRID_DOWNSAMPLING, shardConf.getNumId()) || gridSec == Service.GRID_ABSENT) {
            for (var request : requests) {
                result.put(request, request);
            }
            return result;
        }

        long shardGridMillis = TimeUnit.MILLISECONDS.toMillis(gridSec);
        final long roundGridMillis;
        if (shardGridMillis > globalGridMillis) {
            roundGridMillis = shardGridMillis;
        } else {
            roundGridMillis = InstantUtils.truncate(globalGridMillis, shardGridMillis);
        }

        for (var request : requests) {
            // round only auto grid, current implementation ugly because downsampling
            // it's not separate parameter outside expression
            if (request.getGridMillis() != 0 && request.getGridMillis() == globalGridMillis) {
                result.put(request, request.toBuilder()
                        .setGridMillis(roundGridMillis)
                        .build());
            } else {
                result.put(request, request);
            }
        }

        return result;
    }

    private PreparedProgram prepareProgram(ParsedTarget target, Interval interval) {
        Program p = Program.fromSource("").withExternalExpression(target.expression).withDeprOpts(DeprOpts.GRAFANA).compile();
        return p.prepare(interval);
    }

    private CompletableFuture<List<ResolvedMetrics>> resolveMetrics(
            Collection<GraphDataLoadRequest> requests,
            Deadline deadline,
            GrafanaMetrics.DashboardMetrics dashboardMetrics)
    {
        return requests.stream()
            .map(request -> resolveMetrics(request, deadline))
            .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(resolvedMetrics -> {
                int countMetrics = resolvedMetrics.stream()
                    .mapToInt(value -> value.getMetrics().size())
                    .sum();
                dashboardMetrics.readMetrics.add(countMetrics);
                if (countMetrics > DataLimits.MAX_METRICS_FOR_AGGR_COUNT) {
                    throw new BadRequestException("Too many metrics requested. Maximum allowed is " + DataLimits.MAX_METRICS_FOR_AGGR_COUNT +
                        ". Found " + countMetrics + " metric for selectors: " + requests
                        .stream()
                        .map(request -> request.getSelectors().toString())
                        .distinct()
                        .collect(Collectors.joining("; ")));
                }

                deadline.check();
                return resolvedMetrics;
            });
    }

    private CompletableFuture<ResolvedMetrics> resolveMetrics(GraphDataLoadRequest request, Deadline deadline) {
        Selectors selectors = request.getSelectors();
        Selectors selectorsWithoutShard = ShardSelectors.withoutShardKey(selectors);
        ShardKey shardKey = ShardSelectors.getShardKeyOrNull(selectors);

        if (shardKey == null || selectorsWithoutShard.isEmpty()) {
            return CompletableFuture.completedFuture(new ResolvedMetrics(request, Collections.emptyList()));
        }

        return metaSearch.search(shardKey, selectorsWithoutShard, DataLimits.MAX_METRICS_FOR_AGGR_COUNT, "", deadline)
            .thenApply(searchResult -> {
                    if (searchResult.isTruncated()) {
                        throw new BadRequestException("Too many metrics requested. Maximum allowed is " + DataLimits.MAX_METRICS_FOR_AGGR_COUNT +
                            ". Found more metric than that for selector \"" + request.getSelectors() + "\"");
                    }

                List<String> rawNonTrivialLabels = PageMetricStatsUtils.build(searchResult.getLabelsStream());

                Optional<Labels> metricLabels = searchResult.getLabelsStream().findAny();

                List<String> namingLabels =
                        WwwLabelUtils.getNamingLabels(rawNonTrivialLabels, metricLabels, selectorsWithoutShard);

                    return searchResult.getMetricsStream()
                        .map(s -> new Metric(s.getKey(), nameForMetric(s.getLabels(), namingLabels), s.getLabels()))
                        .sorted(Comparator.comparing(Metric::getName, TextComparator.compareTextWithNumbers()))
                        .collect(collectingAndThen(toList(), metrics -> new ResolvedMetrics(request, metrics)));
                }
            );
    }

    private CompletableFuture<GraphDataLoader> loadMetrics(
        String shardId,
        List<ResolvedMetrics> resolved,
        Deadline deadline,
        String subjectId)
    {
        return resolved.stream()
            .map(metrics -> loadMetrics(shardId, metrics, deadline, subjectId))
            .collect(collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(ExpGraphDataLoader::new);
    }

    private CompletableFuture<LoadedMetrics> loadMetrics(String shardId, ResolvedMetrics resolved, Deadline deadline, String subjectId) {
        ReadMetrics.ShardMetrics shardMetrics = clientMetrics.getShardMetrics(shardId);
        List<MetricStorageRequest> loadRequests = resolved.getMetrics()
            .stream()
            .map(s -> createGraphDataRequest(s.getKey(), resolved.getRequest()))
            .collect(toList());

        int loadRequestsCount = loadRequests.size();
        shardMetrics.addReadInterval(loadRequestsCount, resolved.getRequest().getInterval());
        final int countReads = loadRequestsCount;
        long responseSizeBytes = TimeSeriesSizeCounter.estimatedLoadSizeInBytes(
            loadRequestsCount,
            resolved.getRequest().getInterval(),
            resolved.getRequest().getGridMillis());

        if (responseSizeBytes > DataLimits.MAX_METRIC_LINES_IN_BYTES) {
            String message =
                "Memory limit for graph data response is reached, please decrease lines count or interval duration: "
                    + "metric requests count - " + loadRequestsCount
                    + ", interval duration - " + resolved.getRequest().getInterval() + " ms"
                    + ", downsampling grid - " + resolved.getRequest().getGridMillis() + " ms"
                    + ", response size - " + responseSizeBytes + " bytes"
                    + " > " + DataLimits.MAX_METRIC_LINES_IN_BYTES + " bytes";
            return CompletableFuture.failedFuture(new BadRequestException(message));
        }

        long startNanoTime = shardMetrics.started(countReads);
        return graphDataClient.fetch(loadRequests, null, deadline, subjectId)
            .thenApply(results -> {
                List<Metric> metrics = resolved.getMetrics();
                List<LoadedMetric> loaded = IntStream.range(0, metrics.size())
                    .mapToObj(index -> new LoadedMetric(metrics.get(index), MetricArchiveImmutable
                        .of(results.get(index).graphData)))
                    .collect(toList());

                return new LoadedMetrics(resolved.getRequest(), loaded);
            })
            .whenComplete((ignore, ignore2) -> shardMetrics.completed(countReads, startNanoTime));
    }

    private MetricStorageRequest createGraphDataRequest(MetricKey key, GraphDataLoadRequest request) {
        AggregateFunctionType fn = request.getAggregateFunction() != null
            ? request.getAggregateFunction()
            : AggregateFunctionType.DEFAULT;
        return new MetricStorageRequest(
            request.getInterval(),
            key,
            request.getGridMillis(),
            fn,
            OperationDownsampling.FillOption.NULL,
            false,
            request.getRankFilter());
    }

    private String nameForMetric(Labels labels, List<String> namingLabels) {
        if (namingLabels.size() == 1) {
            String key = namingLabels.get(0);
            Label label = labels.findByKey(key);
            if (label == null) {
                return LabelValues.ABSENT;
            }
            return label.getValue();
        } else {
            return FormatLabel.format(labels.stream()
                .filter(l -> namingLabels.contains(l.getKey())));
        }
    }

    private static class ParsedTarget {
        private final String salmonId;
        private final String expression;

        private ParsedTarget(String shardId, String expression) {
            this.salmonId = shardId;
            this.expression = expression;
        }
    }

    @ParametersAreNonnullByDefault
    private static class PatchedGraphDataLoader implements GraphDataLoader {
        private final GraphDataLoader loader;
        private final Map<GraphDataLoadRequest, GraphDataLoadRequest> updatedByOriginal;

        public PatchedGraphDataLoader(GraphDataLoader loader, Map<GraphDataLoadRequest, GraphDataLoadRequest> updatedByOriginal) {
            this.loader = loader;
            this.updatedByOriginal = updatedByOriginal;
        }

        @Override
        public NamedGraphData[] loadGraphData(GraphDataLoadRequest request) {
            return loader.loadGraphData(updatedByOriginal.get(request));
        }
    }
}
