package ru.yandex.solomon.gateway.api.old.grafana;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.backend.meta.search.MetaSearch;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValues;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.selector.LabelSelectorSet;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.combined.DataLimits;
import ru.yandex.solomon.util.time.Deadline;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class GrafanaMetaDataManager {
    private final SolomonConfHolder confHolder;
    private final MetaSearch metaSearch;
    private final Authorizer authorizer;

    @Autowired
    public GrafanaMetaDataManager(
        SolomonConfHolder confHolder,
        MetaSearch metaSearch,
        Authorizer authorizer)
    {
        this.confHolder = confHolder;
        this.metaSearch = metaSearch;
        this.authorizer = authorizer;
    }

    CompletableFuture<List<String>> grafanaListMetricsKey(AuthSubject authSubject, Map<String, String> labels) {
        Deadline deadline = Deadline.fromThreadLocal("grafanaLisTMetricKey");

        if (labels.isEmpty()) {
            return CompletableFuture.completedFuture(List.of(LabelKeys.PROJECT));
        }

        String projectId = labels.getOrDefault(LabelKeys.PROJECT, "");
        if (projectId.isEmpty()) {
            return CompletableFuture.failedFuture(new BadRequestException("cannot show label keys without project"));
        }

        return authorizer.authorize(authSubject, projectId, Permission.METRIC_NAMES_GET)
            .thenCompose(account -> {
                if (!existShardKey(Selectors.of(labels))) {
                    return CompletableFuture.completedFuture(
                        Stream.of(LabelKeys.PROJECT, LabelKeys.CLUSTER, LabelKeys.SERVICE)
                            .filter(key -> !labels.containsKey(key))
                            .sorted()
                            .collect(Collectors.toList()));
                }

                ShardKey shardKey = ShardKey.get(labels);
                return metaSearch.search(shardKey,
                        LabelSelectorSet.fromMap(labels).withoutShardKey().toNewSelectors(),
                        DataLimits.MAX_METRICS_FOR_AGGR_COUNT, null, deadline)
                    .thenApply(
                        searchResult -> searchResult.getLabelsStream()
                            .map(labelList -> labelList.addAll(shardKey.toLabels()))
                            .flatMap(labelList ->
                                labelList.stream()
                                    .filter(label -> !labels.containsKey(label.getKey()))
                                    .map(Label::getKey)
                            )
                            .distinct()
                            .sorted()
                            .collect(Collectors.toList()));
            });
    }

    CompletableFuture<List<Map<String, String>>> grafanaListMetricsValue(AuthSubject authSubject, Map<String, String> labels) {
        Deadline deadline = Deadline.fromThreadLocal("grafanaListMetricsValue");

        return listMetricsValues(authSubject, Selectors.of(labels), deadline);
    }

    CompletableFuture<List<Map<String, String>>> grafanaMetricsSearch(AuthSubject authSubject, String selectorsText) {
        Deadline deadline = Deadline.fromThreadLocal("grafanaMetricsSearch");

        return listMetricsValues(authSubject, Selectors.parse(selectorsText), deadline);
    }

    private CompletableFuture<List<Map<String, String>>> listMetricsValues(AuthSubject authSubject, Selectors selectors, Deadline deadline) {
        String projectId = getOrDefault(selectors, LabelKeys.PROJECT, "");

        if (projectId.isEmpty()) {
            return CompletableFuture.failedFuture(new BadRequestException("cannot show label values without project"));
        }

        boolean showProjectsOnly = LabelValues.ANY.equals(projectId) && selectors.size() == 1;
        if (showProjectsOnly) {
            return grafanaListProjectIds();
        }

        return authorizer.authorize(authSubject, projectId, Permission.METRIC_LABELS_GET)
            .thenCompose(account -> {
                String project = getOrDefault(selectors, LabelKeys.PROJECT, null);
                String cluster = getOrDefault(selectors, LabelKeys.CLUSTER, null);
                String service = getOrDefault(selectors, LabelKeys.SERVICE, null);

                if (!existShardKey(selectors)) {
                    return grafanaListShardKeyValues(project, cluster, service);
                }

                return grafanaListMetricsValueImpl(selectors, deadline);
            });
    }

    private CompletableFuture<List<Map<String, String>>> grafanaListProjectIds() {
        SolomonConfWithContext solomonConf = confHolder.getConfOrThrow();
        List<Map<String, String>> result = solomonConf.getShardKeys().stream()
            .map(ShardKey::getProject)
            .distinct()
            .map(project -> Map.of(LabelKeys.PROJECT, project))
            .collect(Collectors.toList());
        return CompletableFuture.completedFuture(result);
    }

    private static boolean existShardKey(Selectors selectors) {
        String project = getOrDefault(selectors, LabelKeys.PROJECT, LabelValues.ANY);
        if (LabelValues.ANY.equals(project)) {
            return false;
        }

        String cluster = getOrDefault(selectors, LabelKeys.CLUSTER, LabelValues.ANY);
        if (LabelValues.ANY.equals(cluster)) {
            return false;
        }

        String service = getOrDefault(selectors, LabelKeys.SERVICE, LabelValues.ANY);
        if (LabelValues.ANY.equals(service)) {
            return false;
        }

        return true;
    }

    private CompletableFuture<List<Map<String, String>>> grafanaListShardKeyValues(
        @Nullable String project,
        @Nullable String cluster,
        @Nullable String service)
    {
        SolomonConfWithContext solomonConf = confHolder.getConfOrThrow();
        List<Map<String, String>> result = solomonConf.getShardKeys().stream()
            .filter(shardKey -> project == null || project.equals(LabelValues.ANY) || project.equals(shardKey.getProject()))
            .filter(shardKey -> cluster == null || cluster.equals(LabelValues.ANY) || cluster.equals(shardKey.getCluster()))
            .filter(shardKey -> service == null || service.equals(LabelValues.ANY) || service.equals(shardKey.getService()))
            .map(shardKey -> {
                LabelsBuilder builder = Labels.builder(2);
                if (project != null) {
                    builder.add(LabelKeys.PROJECT, shardKey.getProject());
                }
                if (cluster != null) {
                    builder.add(LabelKeys.CLUSTER, shardKey.getCluster());
                }
                if (service != null) {
                    builder.add(LabelKeys.SERVICE, shardKey.getService());
                }
                return builder.build();
            })
            .distinct()
            .map(Labels::toMap)
            .collect(Collectors.toList());

        return CompletableFuture.completedFuture(result);
    }

    private CompletableFuture<List<Map<String, String>>> grafanaListMetricsValueImpl(Selectors selectors, Deadline deadline) {
        ShardKey shardKey = ShardSelectors.getShardKeyOrNull(selectors);
        if (shardKey == null) {
            return CompletableFuture.failedFuture(new BadRequestException("unknown shard key for selectors " + Selectors.format(selectors)));
        }
        return metaSearch.search(shardKey, ShardSelectors.withoutShardKey(selectors),
                DataLimits.MAX_METRICS_FOR_AGGR_COUNT, null, deadline)
            .thenApply(searchResult -> searchResult.getLabelsStream()
                .map(labelList -> filterLabels(labelList, selectors))
                .map(labelList -> labelList.addAll(shardKey.toLabels()))
                .distinct()
                .map(Labels::toMap)
                .collect(Collectors.toList())
            );
    }

    private static Labels filterLabels(
        Labels labelList, Selectors selectors)
    {
        LabelsBuilder builder = Labels.builder(selectors.size());
        labelList.forEach(l -> {
            if (selectors.hasKey(l.getKey())) {
                builder.add(l);
            }
        });
        return builder.build();
    }

    private static String getOrDefault(Selectors selectors, String key, String defaultValue) {
        return Optional.ofNullable(selectors.findByKey(key))
                .map(Selector::getValue)
                .orElse(defaultValue);
    }
}
