package ru.yandex.solomon.gateway.api.staffOnly;

import java.util.HashMap;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.passport.tvmauth.NativeTvmClient;
import ru.yandex.passport.tvmauth.TvmApiSettings;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.protobuf.frontend.TStaffOnlyConfig;
import ru.yandex.solomon.config.protobuf.frontend.TStaffOnlyConfig.TServiceConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.spring.ConditionalOnBean;

/**
 * @author Oleg Baryshnikov
 */
@Configuration
@ConditionalOnBean(TStaffOnlyConfig.class)
@ParametersAreNonnullByDefault
public class GlobalStaffOnlyContext {

    @Nullable
    @Bean(name = "staffOnlyTvmClient")
    TvmClient serviceTicketTvmClient(TStaffOnlyConfig staffOnlyConfig, TAuthConfig authConfig, SecretProvider secrets) {
        HashMap<String, Integer> clientIds = new HashMap<>();
        for (TServiceConfig serviceConfig : staffOnlyConfig.getServicesList()) {
            if (serviceConfig.hasAuthTvm()) {
                clientIds.put(serviceConfig.getName(), serviceConfig.getAuthTvm().getDstClientId());
            }
        }

        if (clientIds.isEmpty()) {
            return null;
        }

        Preconditions.checkArgument(authConfig.hasTvmConfig(),
                "AuthConfig.TvmConfig cannot be empty, " +
                "because StaffOnlyConfig has " + clientIds.keySet() +
                " services, which use TVM auth");

        TAuthConfig.TTvmConfig tvmConfig = authConfig.getTvmConfig();
        Optional<String> clientSecret = secrets.getSecret(tvmConfig.getSecret());
        if (clientSecret.isEmpty()) {
            return null;
        }

        return new NativeTvmClient(TvmApiSettings.create()
            .setSelfTvmId(tvmConfig.getClientId())
            .enableServiceTicketChecking()
            .enableServiceTicketsFetchOptions(clientSecret.get(), clientIds));
    }

    @Bean
    AuthProvider tokenProviderFactory(
            @Qualifier("staffOnlyTvmClient") Optional<TvmClient> tvmClient,
            @Qualifier("iamTokenProvider") Optional<TokenProvider> iamTokenProvider)
    {
        return new AuthProvider(tvmClient, iamTokenProvider);
    }

    @Bean
    public Services services(TStaffOnlyConfig config, ThreadPoolProvider threads, AuthProvider authProvider) {
        return new Services(config, threads, authProvider);
    }

}
