package ru.yandex.solomon.gateway.api.staffOnly;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.net.HostAndPort;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.core.io.buffer.DataBufferUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.config.protobuf.frontend.TStaffOnlyConfig;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.util.collection.Nullables;

import static org.springframework.web.bind.annotation.RequestMethod.DELETE;
import static org.springframework.web.bind.annotation.RequestMethod.GET;
import static org.springframework.web.bind.annotation.RequestMethod.POST;
import static org.springframework.web.bind.annotation.RequestMethod.PUT;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(path = "/staffOnly", produces = MediaType.TEXT_HTML_VALUE)
@Import({
    ManagerWriterContext.class,
    GlobalStaffOnlyContext.class,
})
@ConditionalOnBean(TStaffOnlyConfig.class)
@ParametersAreNonnullByDefault
public class GlobalStaffOnlyController {

    private final Services services;
    private final InternalAuthorizer authorizer;

    @Autowired
    public GlobalStaffOnlyController(Services services, InternalAuthorizer authorizer) {
        this.services = services;
        this.authorizer = authorizer;
    }

    @RequestMapping(method = GET)
    public CompletableFuture<String> staffOnlyServices(
        @RequireAuth AuthSubject authSubject,
        @RequestParam Map<String, String> params)
    {
        return authorizer.authorize(authSubject)
            .thenApply(account -> new ServicesWww(services, params).genString());
    }

    @RequestMapping(path = "/{address}/**", method = {GET, POST, PUT, DELETE})
    public CompletableFuture<ResponseEntity<?>> manager(
        ServerHttpRequest serverRequest,
        @RequireAuth AuthSubject authSubject,
        @PathVariable("address") String address,
        @RequestParam Map<String, String> params)
    {
        var service = services.findByAddress(HostAndPort.fromString(address));
        if (service == null) {
            return CompletableFuture.failedFuture(new NotFoundException("unknown address: " + address));
        }

        return authorizer.authorize(authSubject)
            .thenCompose(account -> buildClientRequest(serverRequest, service, address, params))
            .thenCompose(clientRequest -> service.getClient().send(address, clientRequest));
    }

    private CompletableFuture<Request> buildClientRequest(
        ServerHttpRequest serverRequest,
        Services.Service service,
        String address,
        Map<String, String> params)
    {
        String path = prepareManagerPath(service, serverRequest, address);

        RequestBuilder request = new RequestBuilder(serverRequest.getMethodValue());
        request.setUrl("http://" + address + path);
        request.setHeaders(prepareHeaders(serverRequest));

        for (Map.Entry<String, String> entry : params.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            request.addQueryParam(key, value);
        }

        if (serverRequest.getMethod() == HttpMethod.POST || serverRequest.getMethod() == HttpMethod.PUT) {
            return DataBufferUtils.join(serverRequest.getBody())
                .toFuture()
                .thenApply(body -> {
                    request.setBody(body.asByteBuffer());
                    return request.build();
                });
        }

        return CompletableFuture.completedFuture(request.build());
    }

    private static String prepareManagerPath(Services.Service service, ServerHttpRequest request, String address) {
        int prefixLen = "/staffOnly/".length() + address.length();
        String path = request.getPath().toString().substring(prefixLen);
        if (path.isEmpty() || "/".equals(path)) {
            return service.getRootPage();
        }
        return path;
    }

    private static HttpHeaders prepareHeaders(ServerHttpRequest request) {
        HttpHeaders orig = request.getHeaders();
        HttpHeaders result = new HttpHeaders();
        result.addAll(HttpHeaders.CONTENT_TYPE, Nullables.orEmpty(orig.get(HttpHeaders.CONTENT_TYPE)));
        result.addAll(HttpHeaders.REFERER, Nullables.orEmpty(orig.get(HttpHeaders.REFERER)));
        return result;
    }
}
