package ru.yandex.solomon.gateway.api.staffOnly;

import java.util.function.Predicate;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
abstract class LinksRepair {

    private final String replacePattern;

    LinksRepair(String replacePattern) {
        this.replacePattern = replacePattern;
    }

    static LinksRepair regex(String findPattern, String replacePattern) {
        return new RegexRepair(findPattern, replacePattern);
    }

    static LinksRepair simple(String findPattern, String replacePattern) {
        return new SimpleRepair(findPattern, replacePattern);
    }

    static LinksRepair regexpPredicate(String findPattern, String replacePattern, Predicate<MatchResult> predicate) {
        return new RegexPredicateRepair(findPattern, replacePattern, predicate);
    }

    String repair(String content, String address) {
        var replacement = StringUtils.replace(replacePattern, "{{address}}", address);
        return replaceAll(content, replacement);
    }

    abstract String replaceAll(String content, String replacement);

    /**
     * Uses simple linear search to perform replacements.
     */
    @ParametersAreNonnullByDefault
    private static class SimpleRepair extends LinksRepair {
        private final String findPattern;

        SimpleRepair(String findPattern, String replacePattern) {
            super(replacePattern);
            this.findPattern = findPattern;
        }

        @Override
        String replaceAll(String content, String replacement) {
            return StringUtils.replace(content, findPattern, replacement);
        }
    }

    /**
     * Uses regular expression to perform replacements.
     */
    @ParametersAreNonnullByDefault
    private static class RegexRepair extends LinksRepair {
        private final Pattern findPattern;

        RegexRepair(String findPattern, String replacePattern) {
            super(replacePattern);
            this.findPattern = Pattern.compile(findPattern);
        }

        @Override
        String replaceAll(String content, String replacement) {
            Matcher matcher = findPattern.matcher(content);
            return matcher.replaceAll(replacement);
        }
    }

    private static class RegexPredicateRepair extends LinksRepair {
        private final Pattern findPattern;
        private final Predicate<MatchResult> predicate;

        public RegexPredicateRepair(String findPattern, String replacePattern, Predicate<MatchResult> predicate) {
            super(replacePattern);
            this.findPattern = Pattern.compile(findPattern);
            this.predicate = predicate;
        }

        @Override
        String replaceAll(String content, String replacement) {
            Matcher matcher = findPattern.matcher(content);
            return matcher.replaceAll(matchResult -> {
                if (!predicate.test(matchResult)) {
                    return matchResult.group();
                }

                return replacement;
            });
        }
    }
}
