package ru.yandex.solomon.gateway.api.staffOnly;

import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.misc.asyncHttpClient2.AsyncHttpClientUtils;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
final class ServiceHttpClient {

    private static final List<LinksRepair> HTML_LINKS_REPAIRS = List.of(
        LinksRepair.regex("href=\"/(?!staffOnly)", "href=\"/staffOnly/{{address}}/"),
        LinksRepair.regex("href='/(?!staffOnly)", "href='/staffOnly/{{address}}/"),

        LinksRepair.simple("action=\"/", "action=\"/staffOnly/{{address}}/"),
        LinksRepair.simple("action='/", "action='/staffOnly/{{address}}/"),

        LinksRepair.simple("src=\"/", "src=\"/staffOnly/{{address}}/"),
        LinksRepair.simple("src='/", "src='/staffOnly/{{address}}/")
    );

    private static final LinksRepair CSS_LINKS_REPAIR =
        LinksRepair.simple("url(\"/", "url(\"/staffOnly/{{address}}/");

    private final AuthType authType;
    private final TokenProvider tokenProvider;
    private final AsyncHttpClient asyncHttpClient;
    private final LinksRepair fqdnRepair;

    ServiceHttpClient(AuthType authType, TokenProvider tokenProvider, Predicate<String> fqdnPredicate) {
        this.authType = authType;
        this.tokenProvider = tokenProvider;
        this.asyncHttpClient = AsyncHttpClientUtils.newAsyncHttpClient(getClass().getSimpleName(), "solomon-manager");
        this.fqdnRepair = LinksRepair.regexpPredicate("https?://([0-9a-zA-Z\\._-]+:\\d+)", "/staffOnly/$1", matchResult -> {
            return fqdnPredicate.test(matchResult.group(1));
        });
    }

    CompletableFuture<ResponseEntity<?>> send(String address, Request request) {
        if (authType != AuthType.Anonymous) {
            request = new RequestBuilder(request)
                .addHeader(authType.getHeaderName(), authType.getValuePrefix() + tokenProvider.getToken())
                .build();
        }

        final String path = request.getUrl();
        return AsyncHttpClientUtils.execute(asyncHttpClient, request)
            .thenApply(response -> {
                if (isBinaryFile(path)) {
                    return new ResponseEntity<>(
                        response.getResponseBodyAsBytes(),
                        convertHeaders(response.getHeaders()),
                        getHttpStatus(response));
                }
                return mapToResponseEntity(response, address);
            });
    }

    private ResponseEntity<String> mapToResponseEntity(Response response, String address) {
        HttpStatus httpStatus = getHttpStatus(response);
        if (httpStatus == HttpStatus.FOUND) {
            String location = response.getHeader("location");
            if (StringUtils.isNoneBlank(location)) {
                if (location.startsWith("/")) {
                    location = "/staffOnly/" + address + location;
                } else if (location.startsWith("http://")) {
                    location = "/staffOnly/" + StringUtils.removeStart(location, "http://");
                }

                HttpHeaders headers = new HttpHeaders();
                headers.setLocation(URI.create(location));
                return new ResponseEntity<>(headers, HttpStatus.FOUND);
            }
        }

        var contentType = Nullables.orEmpty(response.getContentType());
        var content = response.getResponseBody();

        // server can return charset encoding like this 'text/html;charset=UTF-8',
        // so use startsWith check here
        if (contentType.startsWith("text/html")) {
            for (LinksRepair r : HTML_LINKS_REPAIRS) {
                content = r.repair(content, address);
            }
            content = fqdnRepair.repair(content, address);

        } else if (contentType.startsWith("text/css")) {
            content = CSS_LINKS_REPAIR.repair(content, address);
        }

        return new ResponseEntity<>(content, convertHeaders(response.getHeaders()), httpStatus);
    }

    static boolean isBinaryFile(String path) {
        return path.endsWith(".woff") || path.endsWith(".woff2") || path.endsWith(".png");
    }

    static HttpHeaders convertHeaders(io.netty.handler.codec.http.HttpHeaders headers) {
        var newHeaders = new HttpHeaders();
        for (Map.Entry<String, String> h : headers) {
            newHeaders.add(h.getKey(), h.getValue());
        }
        return newHeaders;
    }

    private static HttpStatus getHttpStatus(Response response) {
        HttpStatus httpStatus;
        try {
            httpStatus = HttpStatus.valueOf(response.getStatusCode());
        } catch (IllegalStateException e) {
            httpStatus = HttpStatus.INTERNAL_SERVER_ERROR;
        }
        return httpStatus;
    }
}
