package ru.yandex.solomon.gateway.api.staffOnly;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.net.HostAndPort;
import com.google.common.util.concurrent.MoreExecutors;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.cluster.discovery.ClusterDiscoveryImpl;
import ru.yandex.discovery.DiscoveryService;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.config.protobuf.frontend.TStaffOnlyConfig;
import ru.yandex.solomon.config.protobuf.frontend.TStaffOnlyConfig.TServiceConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;

/**
 * @author Sergey Polovko
 */
final class Services {

    private final Map<String, Service> services;

    public Services(TStaffOnlyConfig config, ThreadPoolProvider threads, AuthProvider authProvider) {
        var services = new TreeMap<String, Service>();
        for (TServiceConfig serviceConfig : config.getServicesList()) {
            services.put(
                serviceConfig.getName(),
                new Service(serviceConfig, threads.getSchedulerExecutorService(), authProvider, this::hasAddress));
        }
        this.services = Collections.unmodifiableMap(services);
    }

    public Collection<Service> getAll() {
        return services.values();
    }

    @Nullable
    public Service get(String name) {
        return services.get(name);
    }

    public boolean hasAddress(String address) {
        try {
            return findByAddress(HostAndPort.fromString(address)) != null;
        } catch (Throwable e) {
            return false;
        }
    }

    @Nullable
    public Service findByAddress(HostAndPort address) {
        for (Service service : services.values()) {
            if (service.contains(address)) {
                return service;
            }
        }
        return null;
    }

    public static final class Service {
        private final String name;
        private final String rootPage;
        private final ClusterDiscovery<FakeTransport> discovery;
        private final ServiceHttpClient client;

        public Service(TServiceConfig config, ScheduledExecutorService timer, AuthProvider authProvider, Predicate<String> fqdnPredicate) {
            this.name = config.getName();
            this.rootPage = config.getRootPage();
            this.discovery = new ClusterDiscoveryImpl<>(
                FakeTransport::new,
                config.getAddressesList(),
                DiscoveryService.async(),
                timer,
                MoreExecutors.directExecutor(),
                TimeUnit.HOURS.toMillis(1L));

            AuthType authType = authProvider.authType(config);
            TokenProvider tokenProvider = authProvider.tokenProvider(config);
            this.client = new ServiceHttpClient(authType, tokenProvider, fqdnPredicate);
        }

        public String getName() {
            return name;
        }

        public String getRootPage() {
            return rootPage.isEmpty() ? "/internal" : rootPage;
        }

        public ServiceHttpClient getClient() {
            return client;
        }

        public Set<String> getAddresses() {
            return discovery.getNodes()
                    .stream()
                    .map(s -> discovery.getTransportByNode(s).address.toString())
                    .collect(Collectors.toCollection(TreeSet::new));
        }

        public Set<String> getAddresses(String search) {
            return discovery.getNodes()
                .stream()
                .map(s -> discovery.getTransportByNode(s).address.toString())
                .filter(s -> s.contains(search))
                .collect(Collectors.toCollection(TreeSet::new));
        }

        public boolean contains(HostAndPort address) {
            var fqdn = address.getHost();
            if (!discovery.hasNode(fqdn)) {
                return false;
            }
            return address.equals(discovery.getTransportByNode(fqdn).address);
        }
    }

    private static class FakeTransport implements AutoCloseable {
        private final HostAndPort address;

        public FakeTransport(HostAndPort address) {
            this.address = address;
        }

        @Override
        public void close() {
        }
    }
}
