package ru.yandex.solomon.gateway.api.staffOnly;

import java.util.Collection;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.solomon.gateway.api.staffOnly.Services.Service;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.Numbers;

/**
 * @author Sergey Polovko
 */
final class ServicesWww extends ManagerPageTemplate {

    private static final Pattern DC_REGEXP = Pattern.compile("[a-z0-9-]+-(sas|vla|man|myt|iva)-[0-9]+[a-z0-9-.]*(:[0-9]+)?");
    private static final int DEFAULT_LIMIT = 30;

    private static final String SERVICE_PARAM = "service";
    private static final String HOST_PARAM =  "host";
    private static final String LIMIT_PARAM = "limit";

    private final Services services;
    private final Map<String, String> params;

    ServicesWww(Services services, Map<String, String> params) {
        super("Solomon Services", true);
        this.services = services;
        this.params = params;
    }

    @Override
    protected void content() {
        Collection<Service> servicesList = services.getAll();
        if (servicesList.isEmpty()) {
            tag("div", () -> write("no services configured"));
            return;
        }

        var hostParam = params.getOrDefault(HOST_PARAM, "");

        var activeServiceName = params.containsKey(SERVICE_PARAM)
            ? params.get(SERVICE_PARAM)
            : servicesList.stream().filter(service -> !service.getAddresses(hostParam).isEmpty()).findFirst().map(Service::getName).orElse("");

        int limit = Numbers.parseIntSafe(params.get(LIMIT_PARAM), DEFAULT_LIMIT);

        gridXs(1, () -> {
            tag("ul.nav.nav-pills.nav-stacked", () -> {
                for (Service service : servicesList) {
                    var attrs = service.getName().equals(activeServiceName)
                        ? new Attr[]{Attr.cssClass("active")}
                        : EMPTY_ATTRS;
                    li(() -> servicePageLink(service.getName(), hostParam, limit, service.getName()), attrs);
                }
            });
        });

        gridXs(11, () -> {
            var activeService = services.get(activeServiceName);;
            if (activeService == null) {
                tag("div", () -> write("unknown service " + activeServiceName));
                return;
            }

            var addressesPerDc = groupByDc(activeService.getAddresses(hostParam));
            for (Entry<String, Set<String>> e : addressesPerDc.entrySet()) {
                gridXs(3, () -> {
                    panelNoBody(e.getKey(), () -> {
                        var addresses = e.getValue();
                        div(() -> {
                            String rootPage = activeService.getRootPage();
                            int count = limit;
                            for (String address : addresses) {
                                if (count-- == 0) {
                                    break;
                                }
                                aHref("/staffOnly/" + address + rootPage, Attr.cssClass("list-group-item"), address);
                            }
                        }, Attr.cssClass("list-group"));

                        if (limit > 0) {
                            if (addresses.size() > limit) {
                                servicePageLink(activeServiceName, hostParam, -1, "Show " + (addresses.size() - limit) + " more");
                            }
                        } else if (addresses.size() > DEFAULT_LIMIT) {
                            servicePageLink(activeServiceName, hostParam, DEFAULT_LIMIT, "Show 30 first");
                        }
                    });
                });
            }
        });
    }

    private void servicePageLink(String service, String host, int limit, String title) {
        aHref("/staffOnly?"
                + SERVICE_PARAM + '=' + service + '&'
                + LIMIT_PARAM + '=' + limit + '&'
                + HOST_PARAM + "=" + host, title);
    }

    private static Map<String, Set<String>> groupByDc(Set<String> addresses) {
        var result = new TreeMap<String, Set<String>>();
        for (String address : addresses) {
            Matcher matcher = DC_REGEXP.matcher(address);
            String dc = matcher.matches() ? matcher.group(1) : "unknown-dc";
            result.computeIfAbsent(dc, k -> new TreeSet<>())
                .add(address);
        }
        return result;
    }
}
