package ru.yandex.solomon.gateway.api.staffOnly;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.Numbers;

/**
 * @author Ivan Tsybulin
 */
final class ServicesWww2 extends ManagerPageTemplate {

    private static final Pattern DC_REGEXP = Pattern.compile("[a-z0-9-]+-(sas|vla|man|myt|iva)-[0-9]+\\.mon\\.yandex\\.net(:[0-9]+)?");
    private static final int DEFAULT_LIMIT = 10;

    private static final String SERVICE_PARAM = "service";
    private static final String LIMIT_PARAM = "limit";
    private static final String SEARCH_PARAM = "search";

    private final Services services;
    private final Map<String, String> params;

    ServicesWww2(Services services, Map<String, String> params) {
        super("Solomon Services", true);
        this.services = services;
        this.params = params;
    }

    @Override
    protected void header() {
        tag("div.page-header", () -> {
            tag("span", Attr.style("word-wrap: break-word; font-size: xx-large; width: 70%;"), () -> write(title));
            form(() -> {
                tag("input",
                        new Attr("id", "search"),
                        new Attr("type", "textfield"),
                        new Attr("name", SEARCH_PARAM),
                        new Attr("placeholder", "e.g. solomon-fetcher-sas"),
                        new Attr("value", params.getOrDefault(SEARCH_PARAM, "")),
                        Attr.cssClass("form-control"));
                for (var e : params.entrySet()) {
                    if (!e.getKey().equals(SEARCH_PARAM)) {
                        inputHidden(e.getKey(), e.getValue());
                    }
                }
            }, Attr.style("float: right; width: 30%"));
        }, Attr.style("margin-top: 0px"));
    }

    @Override
    protected void content() {
        Collection<Services.Service> servicesList = services.getAll();
        if (servicesList.isEmpty()) {
            tag("div", () -> write("no services configured"));
            return;
        }

        int limit = Numbers.parseIntSafe(params.get(LIMIT_PARAM), DEFAULT_LIMIT);
        String search = params.getOrDefault(SEARCH_PARAM, "");

        Map<String, Set<String>> addressesByService = servicesList.stream()
                .collect(Collectors.toUnmodifiableMap(
                        Services.Service::getName,
                        service -> service.getAddresses(search)
                ));

        Map<String, Map<String, Set<String>>> groupedByDc = addressesByService.entrySet().stream()
                .collect(Collectors.toUnmodifiableMap(
                        Map.Entry::getKey,
                        e -> groupByDc(e.getValue())
                ));

        var activeServiceName = params.containsKey(SERVICE_PARAM) ?
                params.get(SERVICE_PARAM) :
                servicesList.stream()
                    .filter(service -> addressesByService.get(service.getName()).size() > 0)
                    .findFirst()
                    .orElse(servicesList.iterator().next())
                    .getName();

        tag("ul.nav.nav-pills", new Attr("role", "tablist"), () -> {
            for (Services.Service service : servicesList) {
                String name = service.getName();
                boolean selected = name.equals(activeServiceName);
                tag(selected ? "li.nav-item.active" : "li.nav-item", () -> {
                    tag("a.nav-link#" + name + "-tab", () -> {
                            tag("span", () -> writeRaw(name + "&nbsp"));
                            tag("span.badge.badge-light", () -> write(
                                    groupedByDc.get(name).size() + "/" +
                                    addressesByService.get(name).size())
                            );
                        },
                        new Attr("data-toggle", "tab"),
                        new Attr("href", "#" + name),
                        new Attr("role", "tab"),
                        new Attr("aria-controls", name),
                        new Attr("aria-selected", selected ? "true" : "false")
                    );
                });
            }
        });

        tag("div.tab-content#tabContent", () -> {
            for (Services.Service service : servicesList) {
                String name = service.getName();
                boolean selected = name.equals(activeServiceName);
                tag("div#" + name + ".tab-pane" + (selected ? ".active" : ""), () -> {
                    var addressesPerDc = groupedByDc.get(name);
                    for (Map.Entry<String, Set<String>> e : addressesPerDc.entrySet()) {
                        tag("div.dense", () -> panelNoBody(e.getKey(), () -> {
                            var addresses = e.getValue();
                            ul(() -> {
                                String rootPage = service.getRootPage();
                                int count = limit;
                                for (String address : addresses) {
                                    if (count-- == 0) {
                                        break;
                                    }
                                    li(() -> aHref("/staffOnly/" + address + rootPage, address),
                                            Attr.cssClass("hostname"));
                                }
                            }, Attr.cssClass("hostname"));

                            if (limit > 0) {
                                if (addresses.size() > limit) {
                                    servicePageLink(name, -1, "Show " + (addresses.size() - limit) + " more");
                                }
                            } else if (addresses.size() > DEFAULT_LIMIT) {
                                servicePageLink(name, DEFAULT_LIMIT, "Show " + DEFAULT_LIMIT + " first");
                            }
                        }));
                    }
                }, new Attr("role", "tabpanel"), new Attr("aria-labelledby", name + "-tab"));
            }
        });

        tag("style", () -> writeRaw(STYLE));
    }

    private void servicePageLink(String service, int limit, String title) {
        String url = "/staffOnly?" + SERVICE_PARAM + '=' + service + '&' + LIMIT_PARAM + '=' + limit;
        if (params.containsKey(SEARCH_PARAM)) {
            url = url + "&" + SEARCH_PARAM + "=" + params.get(SEARCH_PARAM);
        }
        aHref(url, title);
    }

    private static Map<String, Set<String>> groupByDc(Set<String> addresses) {
        var result = new TreeMap<String, Set<String>>();
        for (String address : addresses) {
            Matcher matcher = DC_REGEXP.matcher(address);
            String dc = matcher.matches() ? matcher.group(1) : "unknown-dc";
            result.computeIfAbsent(dc, k -> new TreeSet<>())
                .add(address);
        }
        return result;
    }

    private static String STYLE = """
        .dense {
            padding-left: 5px;
            padding-right: 5px;
            padding-top: 5px;
            width: 25%;
            float: left;
        }

        ul.hostname {
            padding-top: 10px;
            padding-left: 20px;
        }
        """;
}
