package ru.yandex.solomon.gateway.api.utils.argResolvers;

import org.springframework.core.MethodParameter;
import org.springframework.web.reactive.BindingContext;
import org.springframework.web.reactive.result.method.HandlerMethodArgumentResolver;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.ydb.page.PageOptions;

/**
 * @author Sergey Polovko
 */
public class PageOptionsArgumentResolver implements HandlerMethodArgumentResolver {

    private static final String PAGE_SIZE_PARAM = "pageSize";
    private static final String PAGE_PARAM = "page";

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return PageOptions.class == parameter.getParameterType();
    }

    @Override
    public Mono<Object> resolveArgument(
            MethodParameter parameter,
            BindingContext bindingContext,
            ServerWebExchange exchange)
    {
        var queryParams = exchange.getRequest().getQueryParams();
        String sizeStr = queryParams.getFirst(PAGE_SIZE_PARAM);
        int size = "all".equalsIgnoreCase(sizeStr)
                ? PageOptions.ALL_SIZE
                : parsePositiveIntParam(sizeStr, PAGE_SIZE_PARAM, PageOptions.DEFAULT_SIZE);

        String pageStr = queryParams.getFirst(PAGE_PARAM);
        int page = parsePositiveIntParam(pageStr, PAGE_PARAM, 0);
        return Mono.just(new PageOptions(size, page));
    }

    private static int parsePositiveIntParam(String strValue, String paramName, int defaultValue) {
        if (StringUtils.isEmpty(strValue)) {
            return defaultValue;
        }

        try {
            int intValue = Integer.parseInt(strValue);
            if (intValue < 0) {
                String message = String.format("value of %s param must be positive, but got: %d", paramName, intValue);
                throw new BadRequestException(message);
            } else if (intValue == 0) {
                return defaultValue;
            }
            return intValue;
        } catch (NumberFormatException e) {
            String message = String.format("cannot parse integer value of %s param from '%s'", paramName, strValue);
            throw new BadRequestException(message);
        }
    }
}
