package ru.yandex.solomon.gateway.api.utils.grpc;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.primitives.Ints;
import io.grpc.Context;
import io.grpc.Contexts;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.springframework.http.HttpHeaders;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class EtagInterceptor implements ServerInterceptor {
    private static final Context.Key<Integer> ETAG_CONTEXT_KEY = Context.key("ETag");
    private static final Metadata.Key<String> ETAG_METADATA_KEY = Metadata.Key.of("ETag", Metadata.ASCII_STRING_MARSHALLER);

    private static final Status INVALID_ETAG_STATUS = Status.INVALID_ARGUMENT.withDescription("invalid ETag format");

    public EtagInterceptor() {

    }

    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(
            ServerCall<ReqT, RespT> call,
            Metadata headers,
            ServerCallHandler<ReqT, RespT> next) {
        String etag = headers.get(ETAG_METADATA_KEY);

        int etagValue = parseEtagHeader(etag);

        Context context =
                Context.current().withValue(ETAG_CONTEXT_KEY, etagValue);

        return Contexts.interceptCall(context, call, headers, next);
    }

    public static int getEtag(Context context) {
        return Nullables.orDefault(ETAG_CONTEXT_KEY.get(context), -1);
    }

    public static int parseEtagHeader(ServerHttpRequest request) {
        String etag = request.getHeaders().getFirst(HttpHeaders.ETAG);
        return parseEtagHeader(etag);
    }

    private static int parseEtagHeader(@Nullable String etag) {
        int etagValue;

        if (etag == null || etag.isEmpty()) {
            etagValue = -1;
        } else {
            //noinspection UnstableApiUsage
            Integer parsedEtag = Ints.tryParse(etag);
            if (parsedEtag == null || parsedEtag < 0) {
                throw new StatusRuntimeException(INVALID_ETAG_STATUS);
            }
            etagValue = parsedEtag;
        }
        return etagValue;
    }
}
