package ru.yandex.solomon.gateway.api.v2;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import springfox.documentation.annotations.ApiIgnore;

import ru.yandex.solomon.agent.protobuf.TRegisterAgentRequest;
import ru.yandex.solomon.agent.protobuf.TRegisterAgentResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.tvm.TvmSubject;
import ru.yandex.solomon.core.db.dao.AgentDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.model.Agent;
import ru.yandex.solomon.core.exceptions.AuthorizationException;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.InternalServerErrorException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.v2.dto.AgentDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.util.http.HttpUtils;
import ru.yandex.solomon.ydb.page.PageOptions;

import static ru.yandex.solomon.auth.AuthType.IAM;
import static ru.yandex.solomon.auth.AuthType.TvmService;

/**
 * @author Oleg Baryshnikov
 */
@ApiIgnore
@RestController
@RequestMapping(path = "/api/v2/agents")
@ParametersAreNonnullByDefault
public class AgentController {
    private final static long IP_CACHE_TTL_MINUTES = 30;
    private volatile int registerDelayMinutes = 5;

    private final Authorizer authorizer;
    private final AgentDao agentDao;
    private final ServiceProvidersDao providersDao;
    private final Cache<String, String> ipAddressToHostnameCache;

    @SuppressWarnings("unused")
    @ManagerMethod
    public void setRegisterDelayMinutes(int registerDelayMinutes) {
        this.registerDelayMinutes = registerDelayMinutes;
    }

    @Autowired
    public AgentController(Authorizer authorizer, AgentDao agentDao, ServiceProvidersDao providersDao) {
        this.authorizer = authorizer;
        this.agentDao = agentDao;
        this.providersDao = providersDao;

        this.ipAddressToHostnameCache = CacheBuilder.newBuilder()
            .expireAfterAccess(IP_CACHE_TTL_MINUTES, TimeUnit.MINUTES)
            .build();
    }

    @ApiOperation(
        value = "register agent",
        notes = "This action registers agent installation and returns request duration in milliseconds."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
    })
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "provider", value = "provider id", dataType = "string"),
        @ApiImplicitParam(paramType = "query", name = "clusterId", value = "cluster id", dataType = "string"),
        @ApiImplicitParam(paramType = "body", name = "body", value = "agent registration information"),
    })
    @RequestMapping(path = "/register", method = RequestMethod.POST, consumes = "application/x-protobuf", produces = "application/x-protobuf")
    public CompletableFuture<byte[]> register(
        ServerHttpRequest request,
        @RequireAuth AuthSubject authSubject,
        @RequestParam("provider") String providerId,
        @RequestBody byte[] body)
    {
        if (authSubject.getAuthType() != IAM && authSubject.getAuthType() != TvmService) {
            throw new AuthenticationException("only IAM and TvmService auth types are supported for now");
        }

        Instant lastSeen = Instant.now();

        TRegisterAgentRequest protoRequest = parseRequest(body);
        String hostname = StringUtils.isNotEmpty(protoRequest.getAddress())
                ? protoRequest.getAddress()
                : getHostnameForIpAddress(HttpUtils.realOrRemoteIp(request));

        Agent agent = fromProto(protoRequest, providerId, hostname, lastSeen);

        return providersDao.read(providerId)
            .thenCompose(serviceProvider -> {
                if (serviceProvider.isEmpty()) {
                    throw new NotFoundException("no provider with id: " + providerId);
                }

                var sp = serviceProvider.get();
                switch (authSubject.getAuthType()) {
                    case IAM:
                        if (!sp.getIamServiceAccountIds().contains(authSubject.getUniqueId())) {
                            throw new AuthorizationException("account id differs from the provider's ones");
                        }
                        break;
                    case TvmService:
                        var clientId = ((TvmSubject.ServiceSubject)authSubject).getClientId();
                        if (!sp.getTvmServiceIds().contains(clientId)) {
                            throw new AuthorizationException("service id differs from the provider's ones");
                        }
                        break;
                    default:
                        throw new AuthenticationException("only IAM and TvmService auth types are supported for now");
                }

                return agentDao.insertOrUpdate(agent);
            })
            .thenApply(aVoid -> {
                int delaySeconds = (int) Duration.ofMinutes(registerDelayMinutes).getSeconds();
                TRegisterAgentResponse response = TRegisterAgentResponse.newBuilder()
                    .setRegisterDelaySeconds(delaySeconds)
                    .build();
                return response.toByteArray();
            });
    }

    @ApiOperation(
        value = "find registered agents by provider id",
        notes = "This action returns registered agents by provider id."
    )
    @ApiResponses({
        @ApiResponse(code = 200, message = "success"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "provider was not found"),
    })
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "provider", value = "provider id", dataType = "string"),
        @ApiImplicitParam(paramType = "query", name = "page", value = "page number (starting from 0)", dataType = "integer", defaultValue = "0"),
        @ApiImplicitParam(paramType = "query", name = "pageSize", value = "page size", dataType = "integer", defaultValue = "30"),
    })
    @RequestMapping(method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
    public CompletableFuture<PagedResultDto<AgentDto>> findByProvider(
        @RequireAuth AuthSubject authSubject,
        @RequestParam("provider") String provider,
        PageOptions pageOptions)
    {
        return authorizer.authorize(authSubject, provider, Permission.CONFIGS_LIST)
            .thenCompose(account -> agentDao.findByProvider(provider, pageOptions))
            .thenApply(response -> PagedResultDto.fromModel(response, AgentDto::fromModel));
    }

    private static Agent fromProto(TRegisterAgentRequest proto, String provider, String hostname, Instant lastSeen) {
        return Agent.newBuilder()
            .setProvider(provider)
            .setDataPort(proto.getDataPort())
            .setManagementPort(proto.getManagementPort())
            .setDescription(proto.getDescription())
            .setVersion(proto.getVersion())
            .setLastSeen(lastSeen)
            .setHostname(hostname)
            .setPullIntervalSeconds(proto.getPullIntervalSeconds())
            .build();
    }

    private String getHostnameForIpAddress(String ipAddress) {
        String hostname = this.ipAddressToHostnameCache.getIfPresent(ipAddress);

        if (hostname == null) {
            try {
                hostname = InetAddress.getByName(ipAddress).getHostName();
            } catch (UnknownHostException e) {
                throw new InternalServerErrorException("could not resolve the hostname value for IP: " + ipAddress);
            }

            this.ipAddressToHostnameCache.put(ipAddress, hostname);
        }

        return hostname;
    }

    private TRegisterAgentRequest parseRequest(@RequestBody byte[] body) {
        try {
            return TRegisterAgentRequest.parseFrom(body);
        } catch (Throwable t) {
            throw new BadRequestException("cannot parse protobuf body");
        }
    }
}
