package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.gateway.endpoint.AlertController;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.gateway.api.cloud.v1.AlertingHistoryReader;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertEvaluationHistoryDto;
import ru.yandex.solomon.gateway.api.v2.dto.ValidationUtils;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Alexey Trushkin
 */
@Api(tags = {"alerting"})
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/alerts", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class AlertEvaluationController {
    private static final String SERVICE_PROVIDER_HEADER = "X-Service-Provider";
    private final Authorizer authorizer;
    private final AlertingHistoryReader historyReader;
    private final AlertApi alertApi;

    @Autowired
    public AlertEvaluationController(
            Authorizer authorizer,
            MetricsClient metricsClient,
            @Qualifier("alertingStatuses") ShardKey alertingStatuses,
            AlertApi alertApi)
    {
        this.authorizer = authorizer;
        this.alertApi = alertApi;
        this.historyReader = new AlertingHistoryReader(metricsClient, alertingStatuses);
    }

    @ApiOperation(value = "Get alert evaluation history", response = AlertEvaluationHistoryDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/{alertId}/state/history", method = RequestMethod.GET)
    public CompletableFuture<AlertEvaluationHistoryDto> readEvaluationHistory(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId,
            @RequestParam("from") Instant from,
            @RequestParam("to") Instant to,
            @RequestParam(value = "gridMillis", defaultValue = "60000") long gridMillis)
    {
        ValidationUtils.validateInterval(from, to);
        var interval = Interval.extendToGrid(new Interval(from, to), gridMillis);
        long deadline = System.currentTimeMillis() + Deadline.DEFAULT_TIMEOUT_MILLIS;
        String subjectId = AuthSubject.getLogin(subject, subject.getUniqueId());

        TReadAlertInterpolatedRequest request = TReadAlertInterpolatedRequest.newBuilder()
                .setAlertId(alertId)
                .setProjectId(projectId)
                .setServiceProvider(serviceProviderId)
                .build();

        return AlertController.authorize(authorizer, subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
                .thenCompose(account -> alertApi.readAlert(request))
                .thenCompose(alertResponse -> {
                    var multi = alertResponse.getInterpolatedAlert().getGroupByLabelsCount() > 0;
                    return historyReader.readEvaluationHistory(projectId, alertId, interval, gridMillis, deadline, subjectId, true, multi);
                });
    }
}
