package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolverNoOp;
import ru.yandex.solomon.core.conf.ClustersManager;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterListItemDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterServiceAssociationDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.ShardListItemDto;
import ru.yandex.solomon.util.net.NetworkValidator;
import ru.yandex.solomon.ydb.page.PageOptions;


/**
 * @author Sergey Polovko
 */
@Api(tags = "clusters")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/clusters", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ClustersManager.class })
public class ClustersController {

    private final ClustersManager clustersManager;
    private final NetworkValidator networkValidator;
    private final Authorizer authorizer;
    private final FolderResolver folderResolver;

    @Autowired
    public ClustersController(
        ClustersManager clustersManager,
        NetworkValidator networkValidator,
        Authorizer authorizer,
        Optional<FolderResolver> folderResolver)
    {
        this.clustersManager = clustersManager;
        this.networkValidator = networkValidator;
        this.authorizer = authorizer;
        this.folderResolver = folderResolver.orElseGet(FolderResolverNoOp::new);
    }

    @ApiOperation(
        value = "list available clusters",
        notes = "This action returns project's clusters if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "page", value = "page number (starting from 0)", dataType = "integer", defaultValue = "0"),
        @ApiImplicitParam(paramType = "query", name = "pageSize", value = "page size", dataType = "integer", defaultValue = "30"),
    })
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<ClusterListItemDto>> getAll(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "text", defaultValue = "") String text,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
            .thenCompose(aVoid -> clustersManager.getClusters(projectId, "", pageOptions, text))
            .thenApply(clusters -> PagedResultDto.fromModel(clusters, ClusterListItemDto::fromModel));
    }

    @ApiOperation(
        value = "create cluster",
        notes = "This action will save cluster document if there is no already existed cluster with given id."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<ClusterDto> createCluster(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestBody ClusterDto cluster)
    {
        Instant now = Instant.now();
        cluster.setCreatedAt(now);
        cluster.setUpdatedAt(now);
        cluster.setCreatedBy(subject.getUniqueId());
        cluster.setUpdatedBy(subject.getUniqueId());
        cluster.setProjectId(projectId);

        cluster.validate(networkValidator);

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_CREATE)
            .thenCombine(resolveFolderId(projectId, cluster.getName()), (acc, folderId) -> folderId)
            .thenCompose(folderId -> clustersManager.createCluster(ClusterDto.toModel(cluster, folderId)))
            .thenApply(ClusterDto::fromModel);
    }

    @ApiOperation(
        value = "read one cluster",
        notes = "This action returns single project's cluster found by given clusterId."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "cluster was not found"),
    })
    @RequestMapping(path = "/{clusterId}", method = RequestMethod.GET)
    CompletableFuture<ClusterDto> getCluster(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("clusterId") String clusterId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> clustersManager.getCluster(projectId, "", clusterId))
            .thenApply(ClusterDto::fromModel);
    }

    @ApiOperation(
        value = "update cluster",
        notes = "This action will update already existed project's cluster with given document."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "cluster was not found"),
    })
    @RequestMapping(path = "/{clusterId}", method = RequestMethod.PUT)
    CompletableFuture<ClusterDto> updateCluster(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("clusterId") String clusterId,
        @RequestBody ClusterDto cluster)
    {
        Instant now = Instant.now();
        cluster.setUpdatedAt(now);
        cluster.setUpdatedBy(subject.getUniqueId());
        cluster.setId(clusterId);
        cluster.setProjectId(projectId);
        cluster.validate(networkValidator);

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
            .thenCombine(resolveFolderId(projectId, cluster.getName()), (acc, folderId) -> folderId)
            .thenCompose(folderId -> clustersManager.updateCluster(ClusterDto.toModel(cluster, folderId)))
            .thenApply(ClusterDto::fromModel);
    }

    @ApiOperation(
        value = "delete cluster",
        notes = "This action will delete already existed cluster."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "cluster was not found"),
    })
    @RequestMapping(path = "/{clusterId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteCluster(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("clusterId") String clusterId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_DELETE)
            .thenCompose(aVoid -> clustersManager.deleteCluster(projectId, "", clusterId));
    }

    @ApiOperation(
        value = "associated services",
        notes = "This action returns services associated with cluster found by given clusterId."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(path = "/{clusterId}/services", method = RequestMethod.GET)
    CompletableFuture<List<ClusterServiceAssociationDto>> getClusterServices(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("clusterId") String clusterId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> clustersManager.getClusterAssociations(projectId, "", clusterId))
            .thenApply(associations -> associations.stream()
                .map(ClusterServiceAssociationDto::fromModel)
                .collect(Collectors.toList()));
    }

    private CompletableFuture<String> resolveFolderId(String projectId, String clusterName) {
        return folderResolver.resolveFolderId(projectId, clusterName);
    }

    @ApiOperation(
            value = "associated shards",
            notes = "This action returns shards associated with cluster found by given clusterId."
    )
    @ApiResponses({
            @ApiResponse(code = 401, message = "authentication error"),
            @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(path = "/{clusterId}/shards", method = RequestMethod.GET)
    CompletableFuture<List<ShardListItemDto>> getClusterShards(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("clusterId") String clusterId,
            @RequestParam(value = "state", required = false) EnumSet<ShardState> state)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                .thenCompose(aVoid -> clustersManager.getClusterShards(projectId, "", clusterId, state))
                .thenApply(shards -> shards.stream()
                        .map(ShardListItemDto::fromModel)
                        .collect(Collectors.toList()));
    }

}
