package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.DashboardsManager;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.api.v2.dto.DashboardDto;
import ru.yandex.solomon.gateway.api.v2.dto.DashboardListItemDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.ydb.page.PageOptions;


/**
 * @author Oleg Baryshnikov
 */
@Api(tags = "dashboards")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/dashboards", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ DashboardsManager.class })
@ParametersAreNonnullByDefault
public class DashboardsController {

    @Autowired
    private DashboardsManager dashboardsManager;
    @Autowired
    private Authorizer authorizer;


    @ApiOperation(
        value = "list available dashboards",
        notes = "This action returns project's dashboards if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "text", value = "text filter by id", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "page", value = "page number (starting from 0)", dataType = "integer", defaultValue = "0"),
        @ApiImplicitParam(paramType = "query", name = "pageSize", value = "page size", dataType = "integer", defaultValue = "30"),
    })
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<DashboardListItemDto>> getAll(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "text", defaultValue = "") String text,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
            .thenCompose(aVoid -> dashboardsManager.getDashboards(projectId, "", pageOptions, text))
            .thenApply(dashboards -> PagedResultDto.fromModel(dashboards, DashboardListItemDto::fromModel));
    }

    @ApiOperation(
        value = "read one dashboard",
        notes = "This action returns single project's dashboard found by given dashboardId."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "dashboard was not found"),
    })
    @RequestMapping(path = "/{dashboardId}", method = RequestMethod.GET)
    CompletableFuture<DashboardDto> getDashboard(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("dashboardId") String dashboardId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> dashboardsManager.getDashboard(projectId, "", dashboardId))
            .thenApply(DashboardDto::fromModel);
    }

    @ApiOperation(
        value = "create dashboard",
        notes = "This action will save dashboard document if there is no already existed dashboard with given id."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<DashboardDto> createDashboard(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestBody DashboardDto dashboard)
    {
        Instant now = Instant.now();
        dashboard.createdAt = now;
        dashboard.updatedAt = now;
        dashboard.createdBy = subject.getUniqueId();
        dashboard.updatedBy = subject.getUniqueId();
        dashboard.projectId = projectId;
        dashboard.generatedId = IdGenerator.generateInternalId();
        dashboard.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_CREATE)
            .thenCompose(aVoid -> dashboardsManager.createDashboard(dashboard.toModel()))
            .thenApply(DashboardDto::fromModel);
    }

    @ApiOperation(
        value = "update dashboard",
        notes = "This action will update already existed project's dashboard with given document."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "dashboard was not found"),
    })
    @RequestMapping(path = "/{dashboardId}", method = RequestMethod.PUT)
    CompletableFuture<DashboardDto> updateDashboard(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("dashboardId") String dashboardId,
        @RequestBody DashboardDto dashboard)
    {
        Instant now = Instant.now();
        dashboard.id = dashboardId;
        dashboard.projectId = projectId;
        dashboard.updatedAt = now;
        dashboard.updatedBy = subject.getUniqueId();
        dashboard.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
            .thenCompose(aVoid -> dashboardsManager.updateDashboard(dashboard.toModel()))
            .thenApply(DashboardDto::fromModel);
    }

    @ApiOperation(
        value = "delete dashboard",
        notes = "This action will delete already existed dashboard."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "dashboard was not found"),
    })
    @RequestMapping(path = "/{dashboardId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteProject(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("dashboardId") String dashboardId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_DELETE)
            .thenCompose(aVoid -> dashboardsManager.deleteDashboard(projectId, "", dashboardId));
    }
}
