package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import springfox.documentation.annotations.ApiIgnore;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.v2.dto.data.DataRequestDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.DataResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.DownsamplingDto;
import ru.yandex.solomon.gateway.data.DataClient;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.util.time.Deadline;


/**
 * @author Oleg Baryshnikov
 */
@Api(tags = "data")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/sensors/data", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class DataController {
    private final DataClient dataClient;
    private final ProjectsDao projectsDao;
    private final SolomonConfHolder confHolder;
    private final Authorizer authorizer;

    @Autowired
    public DataController(
        DataClient dataClient,
        ProjectsDao projectsDao,
        SolomonConfHolder confHolder,
        Authorizer authorizer)
    {
        this.dataClient = dataClient;
        this.projectsDao = projectsDao;
        this.confHolder = confHolder;
        this.authorizer = authorizer;
    }

    @ApiIgnore
    @RequestMapping(method = RequestMethod.POST, consumes = MediaType.TEXT_PLAIN_VALUE)
    CompletableFuture<DataResultDto> readDataFromText(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam("from") Instant from,
        @RequestParam("to") Instant to,
        @RequestParam(value = "points", defaultValue = "0") int pointsParam,
        @RequestParam(value = "maxPoints", defaultValue = "0") int maxPointsParam,
        @RequestParam(value = "gridMillis", defaultValue = "0") long gridMillis,
        @RequestParam(value = "downsamplingAggregation", defaultValue = "DEFAULT_AGGREGATION") Aggregation downsamplingAggr,
        @RequestParam(value = "downsamplingFill", defaultValue = "NULL") OperationDownsampling.FillOption downsamplingFill,
        @RequestParam(value = "forceCluster", defaultValue = "", required = false) String forceCluster,
        @Nullable @RequestParam(value = "useNewFormat", required = false) Boolean useNewFormatParam,
        @RequestBody String program)
    {
        Instant deadline = Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);

        DataRequestDto dto = new DataRequestDto();

        dto.program = program;
        dto.from = from;
        dto.to = to;
        dto.downsampling = new DownsamplingDto();
        dto.downsampling.maxPoints = maxPointsParam != 0 ? maxPointsParam : pointsParam;
        dto.downsampling.gridMillis = gridMillis;
        dto.downsampling.aggregation = downsamplingAggr;
        dto.downsampling.fill = downsamplingFill;
        dto.forceCluster = forceCluster;
        dto.useNewFormat = useNewFormatParam;
        dto.validate();

        String subjectId = AuthSubject.getLogin(subject, subject.getUniqueId());
        DataRequest request = dto.toModel(projectId, "", RequestProducer.STAFF, deadline, subjectId);
        return doRead(subject, request);
    }

    @ApiOperation(
        value = "compute metrics data",
        notes = "This action returns data by Solomon-specific program if user have permissions to read that project."
    )
    @ApiResponses({
        @ApiResponse(code = 200, message = "success", response = DataResultDto.class),
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    CompletableFuture<DataResultDto> readDataFromJson(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestBody DataRequestDto requestDto)
    {
        Instant deadline = Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);
        requestDto.validate();
        String subjectId = AuthSubject.getLogin(subject, subject.getUniqueId());
        DataRequest request = requestDto.toModel(projectId, "", RequestProducer.STAFF, deadline, subjectId);
        return doRead(subject, request);
    }

    private CompletableFuture<DataResultDto> doRead(
        AuthSubject subject,
        DataRequest request)
    {
        return authorizer.authorize(subject, request.getProjectId(), Permission.DATA_READ)
            .thenCompose(aVoid -> checkForeignRefs(request.getProjectId()))
            .thenCompose(aVoid -> dataClient.readData(request))
            .thenApply(DataResultDto::fromModel);
    }

    private CompletableFuture<Void> checkForeignRefs(String projectId) {
        SolomonConfWithContext conf = confHolder.getConf();

        Project project = conf == null ? null : conf.getProject(projectId);

        if (project != null) {
            return CompletableFuture.completedFuture(null);
        }

        return projectsDao.exists(projectId)
                .thenAccept(exists -> {
                    if (!exists) {
                        throw new BadRequestException(String.format("project %s does not exist", projectId));
                    }
                });
    }
}
