package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.GraphsManager;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.api.v2.dto.GraphDto;
import ru.yandex.solomon.gateway.api.v2.dto.GraphListItemDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.ydb.page.PageOptions;


/**
 * @author Oleg Baryshnikov
 */
@Api(tags = "graphs")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/graphs", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ GraphsManager.class })
@ParametersAreNonnullByDefault
public class GraphsController {

    @Autowired
    private GraphsManager graphsManager;
    @Autowired
    private Authorizer authorizer;

    @ApiOperation(
        value = "list available graphs",
        notes = "This action returns project's graphs if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "text", value = "text filter by id", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "page", value = "page number (starting from 0)", dataType = "integer", defaultValue = "0"),
        @ApiImplicitParam(paramType = "query", name = "pageSize", value = "page size", dataType = "integer", defaultValue = "30"),
    })
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<GraphListItemDto>> getAll(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "text", defaultValue = "") String text,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
            .thenCompose(aVoid -> graphsManager.getGraphs(projectId, "", pageOptions, text))
            .thenApply(graphs -> PagedResultDto.fromModel(graphs, GraphListItemDto::fromModel));
    }

    @ApiOperation(
        value = "read one graph",
        notes = "This action returns single project's graph found by given graphId."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "graph was not found"),
    })
    @RequestMapping(path = "/{graphId}", method = RequestMethod.GET)
    CompletableFuture<GraphDto> getGraph(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("graphId") String graphId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> graphsManager.getGraph(projectId, "", graphId))
            .thenApply(GraphDto::fromModel);
    }

    @ApiOperation(
        value = "create graph",
        notes = "This action will save graph document if there is no already existed graph with given id."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<GraphDto> createGraph(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestBody GraphDto graph)
    {
        Instant now = Instant.now();
        graph.setCreatedAt(now);
        graph.setUpdatedAt(now);
        graph.setCreatedBy(subject.getUniqueId());
        graph.setUpdatedBy(subject.getUniqueId());
        graph.setProjectId(projectId);
        graph.setGeneratedId(IdGenerator.generateInternalId());
        graph.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_CREATE)
            .thenCompose(aVoid -> graphsManager.createGraph(graph.toModel()))
            .thenApply(GraphDto::fromModel);
    }

    @ApiOperation(
        value = "update graph",
        notes = "This action will update already existed project's graph with given document."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "graph was not found"),
    })
    @RequestMapping(path = "/{graphId}", method = RequestMethod.PUT)
    CompletableFuture<GraphDto> updateGraph(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("graphId") String graphId,
        @RequestBody GraphDto graph)
    {
        Instant now = Instant.now();
        graph.setUpdatedAt(now);
        graph.setUpdatedBy(subject.getUniqueId());
        graph.setId(graphId);
        graph.setProjectId(projectId);
        graph.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
            .thenCompose(aVoid -> graphsManager.updateGraph(graph.toModel()))
            .thenApply(GraphDto::fromModel);
    }

    @ApiOperation(
        value = "delete graph",
        notes = "This action will delete already existed graph."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "graph was not found"),
    })
    @RequestMapping(path = "/{graphId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteGraph(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("graphId") String graphId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_DELETE)
            .thenCompose(aVoid -> graphsManager.deleteGraph(projectId, "", graphId));
    }
}
