package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayMigrationConfig;
import ru.yandex.solomon.core.exceptions.MethodNotAllowedException;
import ru.yandex.solomon.gateway.api.v2.dto.LabelNamesResponseDto;
import ru.yandex.solomon.gateway.api.v2.dto.LabelValuesResponseDto;
import ru.yandex.solomon.gateway.api.v2.dto.MetricDto;
import ru.yandex.solomon.gateway.api.v2.dto.MetricNamesResponseDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.ValidationUtils;
import ru.yandex.solomon.gateway.api.v2.managers.MetricsManager;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;
import ru.yandex.solomon.metrics.client.combined.DataLimits;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.ydb.page.PageOptions;

/**
 * @author Oleg Baryshnikov
 */
@Api(tags = "metrics")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/sensors", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ MetricsManager.class})
@ParametersAreNonnullByDefault
public class MetricsController {

    private final MetricsManager metricsManager;
    private final Authorizer authorizer;
    private final TGatewayMigrationConfig migrationConfig;

    @Autowired
    public MetricsController(
        MetricsManager metricsManager,
        Authorizer authorizer,
        Optional<TGatewayMigrationConfig> migrationConfig)
    {
        this.metricsManager = metricsManager;
        this.authorizer = authorizer;
        this.migrationConfig = migrationConfig.orElse(TGatewayMigrationConfig.getDefaultInstance());
    }

    @ApiOperation(
        value = "find metrics by selector query",
        notes = "This action returns project's metric by selector query if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "selectors", value = "selector query", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "forceCluster", value = "force cluster parameter", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "useNewFormat", value = "use new selector query format", dataType = "boolean"),
    })
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<MetricDto>> findMetrics(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "forceCluster", defaultValue = "") String forceCluster,
        @Nullable @RequestParam(value = "useNewFormat", required = false) Boolean useNewFormatParam,
        PageOptions pageOptions)
    {
        boolean useNewFormat = checkUseOldFormatApi(useNewFormatParam);

        Selectors selectors = Selectors.parse(selectorsStr).toBuilder()
            .addOverride(Selector.exact(LabelKeys.PROJECT, projectId))
            .build();

        return authorizer.authorize(subject, projectId, Permission.METRICS_GET)
            .thenCompose(aVoid -> metricsManager.findMetrics(selectors, forceCluster, pageOptions, useNewFormat))
            .thenApply(metrics -> PagedResultDto.fromModel(metrics, MetricDto::fromModel));
    }

    @ApiOperation(
        value = "find all metric names in project",
        notes = "This action returns metric names in project if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "selectors", value = "selector query", dataType = "string"),
        @ApiImplicitParam(paramType = "query", name = "text", value = "case insensitive text filter by labels", dataType = "string"),
        @ApiImplicitParam(paramType = "query", name = "_validationFilter", value = "label validation filter (internal parameter, will be removed)", dataType = "string", defaultValue = "ALL"),
        @ApiImplicitParam(paramType = "query", name = "limit", value = "metric names count limit [0; " + DataLimits.MAX_METRIC_NAMES_COUNT + "]", dataType = "integer", defaultValue = "0"),
        @ApiImplicitParam(paramType = "query", name = "forceCluster", value = "force cluster parameter (dc abbreviation or empty)", dataType = "string"),
    })
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(path = "/sensorNames", method = RequestMethod.GET)
    CompletableFuture<MetricNamesResponseDto> findMetricNames(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "_validationFilter", defaultValue = "ALL") LabelValidationFilter validationFilter,
        @RequestParam(value = "limit", defaultValue = "0") int limit,
        @RequestParam(value = "forceCluster", defaultValue = "") String forceCluster)
    {
        Instant deadline = deadline();

        Selectors selectors = Selectors.parse(selectorsStr).toBuilder()
            .addOverride(Selector.exact(LabelKeys.PROJECT, projectId))
            .build();

        return authorizer.authorize(subject, projectId, Permission.METRIC_LABELS_GET)
            .thenCompose(aVoid -> metricsManager
                .findMetricNames(selectors, text, validationFilter, limit, forceCluster, deadline))
            .thenApply(MetricNamesResponseDto::fromModel);
    }

    @ApiOperation(
        value = "find all label values by selector query",
        notes = "This action returns values of label by selector query if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "selectors", value = "selector query", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "names", value = "list of required label names joining by comma, leave it empty to show all label names", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "text", value = "case insensitive text filter by labels", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "_validationFilter", value = "label validation filter (internal parameter, will be removed)", dataType = "string", defaultValue = "ALL"),
        @ApiImplicitParam(paramType = "query", name = "limit", value = "label values count limit [0; " + DataLimits.MAX_LABEL_VALUES_COUNT + "]", dataType = "integer", defaultValue = "200"),
        @ApiImplicitParam(paramType = "query", name = "forceCluster", value = "force cluster parameter (dc abbreviation or empty)", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "useNewFormat", value = "use new selector query format", dataType = "boolean"),
    })
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(path = "/labels", method = RequestMethod.GET)
    CompletableFuture<LabelValuesResponseDto> findAllLabelValuesInOldFormat(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "selectors", defaultValue = "") String selectors,
        @RequestParam(value = "names", defaultValue = "") List<String> names,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "_validationFilter", defaultValue = "ALL") LabelValidationFilter validationFilter,
        @RequestParam(value = "limit", defaultValue = "1000") int limit,
        @RequestParam(value = "forceCluster", defaultValue = "") String forceCluster,
        @Nullable @RequestParam(value = "useNewFormat", required = false) Boolean useNewFormatParam)
    {
        Instant deadline = deadline();

        boolean useNewFormat = checkUseOldFormatApi(useNewFormatParam);

        ValidationUtils.validateLabelNames(names);

        Selectors parsedSelectors = Selectors.parse(selectors)
            .toBuilder()
            .addOverride(Selector.exact(LabelKeys.PROJECT, projectId))
            .build();

        return authorizer.authorize(subject, projectId, Permission.METRIC_LABELS_GET)
            .thenCompose(aVoid -> metricsManager
                .findLabelValues(names, parsedSelectors, text, validationFilter, limit, useNewFormat, forceCluster, deadline))
            .thenApply(model -> LabelValuesResponseDto.fromModel(model, parsedSelectors, validationFilter));
    }

    @ApiOperation(
        value = "find label names by selector query",
        notes = "This action returns label names by selector query if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "selectors", value = "selector query", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "forceCluster", value = "force cluster parameter (dc abbreviation or empty)", dataType = "string", defaultValue = ""),
        @ApiImplicitParam(paramType = "query", name = "useNewFormat", value = "use new selector query format", dataType = "boolean"),
    })
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(path = "/names", method = RequestMethod.GET)
    CompletableFuture<LabelNamesResponseDto> findLabelKeysInOldFormat(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "forceCluster", defaultValue = "") String forceCluster,
        @RequestParam(value = "useNewFormat", required = false) Boolean useNewFormatParam)
    {
        Instant deadline = deadline();

        boolean useNewFormat = checkUseOldFormatApi(useNewFormatParam);

        Selectors selectors = Selectors.parse(selectorsStr)
            .toBuilder()
            .addOverride(Selector.exact(LabelKeys.PROJECT, projectId))
            .build();

        return authorizer.authorize(subject, projectId, Permission.METRIC_NAMES_GET)
            .thenCompose(aVoid -> metricsManager.findLabelNames(selectors, useNewFormat, forceCluster, deadline))
            .thenApply(LabelNamesResponseDto::fromModel);
    }

    private boolean checkUseOldFormatApi(@Nullable Boolean useNewFormat) {
        if (useNewFormat == null) {
            return migrationConfig.getForbidOldFormatApiV2();
        }

        if (useNewFormat) {
            return true;
        }

        if (migrationConfig.getForbidOldFormatApiV2()) {
            throw new MethodNotAllowedException("endpoint is not allowed, use /api/v3/projects/:projectId/sensors instead");
        }

        return false;
    }

    private static Instant deadline() {
        return Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);
    }
}
