package ru.yandex.solomon.gateway.api.v2;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.CancelDeleteMetricsOperationRequest;
import ru.yandex.monitoring.api.v3.CreateDeleteMetricsOperationRequest;
import ru.yandex.monitoring.api.v3.GetDeleteMetricsOperationRequest;
import ru.yandex.monitoring.api.v3.ListDeleteMetricsOperationsRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.v2.dto.DeleteMetricsOperationDto;
import ru.yandex.solomon.gateway.api.v2.dto.DeleteMetricsOperationsListDto;
import ru.yandex.solomon.gateway.api.v2.dto.DeleteMetricsRequestDto;
import ru.yandex.solomon.gateway.api.v2.dto.DeleteMetricsResponseDto;
import ru.yandex.solomon.gateway.api.v3.intranet.MetricsDeletionService;

/**
 * @author Stanislav Kashirin
 */
@RestController
@Api(tags = "metrics deletion")
@RequestMapping(path = "/api/v2/projects/{projectId}/sensors/deletion", produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class MetricsDeletionController {
    private final MetricsDeletionService service;

    @Autowired
    public MetricsDeletionController(MetricsDeletionService service) {
        this.service = service;
    }

    @ApiOperation(
        value = "Creates and starts a new metrics deletion operation.",
        notes = "Deletion is bound to the specified project. " +
            "Metrics to delete are resolved by provided selectors. " +
            "Only single shard deletions are supported."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 429, message = "rate limit error"),
    })
    @PostMapping
    CompletableFuture<DeleteMetricsResponseDto> create(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestBody DeleteMetricsRequestDto request)
    {
        request.validate();
        var rq = CreateDeleteMetricsOperationRequest.newBuilder()
            .setProjectId(projectId)
            .setSelectors(request.selectors)
            .setDescription(Strings.nullToEmpty(request.description))
            .build();

        return service.create(rq, subject)
            .thenApply(DeleteMetricsResponseDto::fromProto);
    }

    @ApiOperation(
        value = "Gets the metrics deletion operation by id.",
        notes = "This action returns single metrics deletion operation found by given operationId."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "operation was not found"),
    })
    @GetMapping(path = "/{operationId}")
    CompletableFuture<DeleteMetricsOperationDto> get(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("operationId") String operationId)
    {
        var request = GetDeleteMetricsOperationRequest.newBuilder()
            .setOperationId(operationId)
            .build();

        return service.get(request, subject)
            .thenApply(proto -> {
                if (!proto.getProjectId().equals(projectId)) {
                    throw operationNotFound(projectId, operationId);
                }

                return DeleteMetricsOperationDto.fromProto(proto);
            });
    }

    @ApiOperation(
        value = "Cancels the specified deletion operation by id if it is not too late.",
        notes = "Successful cancellation will interrupt the current deletion " +
            "and will attempt to restore the already deleted metrics.\n\n" +
            "If some of deleted metrics could not be restored, " +
            "the reason will be contained in DeleteMetricsOperationDto.statusMessage. " +
            "Operation will try to restore those metrics until " +
            "the reason will be fixed or permanent deletion time reached. " +
            "In latter case, all non-restored metrics will be deleted permanently.\n\n" +
            "Cancel is an idempotent operation.\n\n" +
            "Cancellation is not possible if permanent deletion has already been launched. " +
            "See DeleteMetricsOperationDto.permanentDeletionAt to determine the point of no return."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "operation was not found"),
    })
    @PutMapping(path = "/{operationId}/cancel")
    CompletableFuture<DeleteMetricsOperationDto> cancel(
        @RequireAuth AuthSubject subject,
        @SuppressWarnings("unused") @PathVariable("projectId") String projectId, // for swagger only
        @PathVariable("operationId") String operationId)
    {
        var request = CancelDeleteMetricsOperationRequest.newBuilder()
            .setOperationId(operationId)
            .build();

        return service.cancelV2(request, subject)
            .thenApply(DeleteMetricsOperationDto::fromProto);
    }

    @ApiOperation(
        value = "Lists all metrics deletion operations in specified project.",
        notes = "This action returns project's deletion operations if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(
            paramType = "query",
            name = "pageSize",
            value = "page size",
            dataType = "integer",
            defaultValue = "100"
        ),
        @ApiImplicitParam(
            paramType = "query",
            name = "pageToken",
            value = "Page token. To get the next page of results, " +
                "set pageToken to the nextPageToken returned by a previous list request.",
            dataType = "string"
        )
    })
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @GetMapping
    CompletableFuture<DeleteMetricsOperationsListDto> list(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
        @RequestParam(value = "pageToken", defaultValue = "") String pageToken)
    {
        var request = ListDeleteMetricsOperationsRequest.newBuilder()
            .setProjectId(projectId)
            .setPageSize(pageSize)
            .setPageToken(pageToken)
            .build();

        return service.list(request, subject)
            .thenApply(DeleteMetricsOperationsListDto::fromProto);
    }

    private static NotFoundException operationNotFound(String projectId, String operationId) {
        return new NotFoundException(
            String.format("no operation with id '%s' in project '%s'", operationId, projectId));
    }
}
