package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ProjectMenuManager;
import ru.yandex.solomon.gateway.api.v2.dto.ProjectMenuDto;


/**
 * @author Oleg Baryshnikov
 */
@Api(tags = "project menus")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/menu", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class ProjectMenuController {

    @Autowired
    private ProjectMenuManager projectMenuManager;
    @Autowired
    private Authorizer authorizer;

    @ApiOperation(
        value = "return project menu",
        notes = "This action returns Solomon frontend menu configuration for project."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "project menu was not found"),
    })
    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<ProjectMenuDto> getProjectMenu(
        @RequireAuth AuthSubject subject,
        @PathVariable(value = "projectId") String projectId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> projectMenuManager.getProjectMenu(projectId))
            .thenApply(ProjectMenuDto::fromModel);
    }

    @ApiOperation(
        value = "save project menu",
        notes = "This action will create or update already existed Solomon frontend menu configuration for project."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "project menu was not found"),
    })
    @RequestMapping(method = { RequestMethod.POST, RequestMethod.PUT })
    CompletableFuture<ProjectMenuDto> saveProjectMenu(
        @RequireAuth AuthSubject subject,
        @PathVariable(value = "projectId") String projectId,
        @RequestBody ProjectMenuDto projectMenu)
    {
        Instant now = Instant.now();
        projectMenu.setId(projectId);
        projectMenu.setCreatedAt(now);
        projectMenu.setUpdatedAt(now);
        projectMenu.setCreatedBy(subject.getUniqueId());
        projectMenu.setUpdatedBy(subject.getUniqueId());
        projectMenu.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
            .thenCompose(aVoid -> projectMenuManager.saveProjectMenu(ProjectMenuDto.toModel(projectMenu)))
            .thenApply(ProjectMenuDto::fromModel);
    }
}
