package ru.yandex.solomon.gateway.api.v2;

import java.util.concurrent.CompletableFuture;

import com.google.common.base.Throwables;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.abc.validator.AbcServiceFieldValidator;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.local.AsUserSubject;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ProjectsManager;
import ru.yandex.solomon.gateway.api.v2.dto.ValidateProjectAvailableDto;
import ru.yandex.solomon.gateway.api.v2.dto.ValidateProjectAvailableResponseDto;

/**
 * @author Alexey Trushkin
 */
@Api(tags = "permissions")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/permissions", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ProjectsManager.class, AbcServiceFieldValidator.class})
public class ProjectPermissionsPublicController {

    private static final String SERVICE_PROVIDER_HEADER = "X-Service-Provider";
    private final Authorizer authorizer;

    @Autowired
    public ProjectPermissionsPublicController(Authorizer authorizer) {
        this.authorizer = authorizer;
    }

    @ApiOperation(
            value = "Check is subject authorized to change project"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "Subject is authorized"),
            @ApiResponse(code = 400, message = "validation error"),
            @ApiResponse(code = 401, message = "authentication error"),
            @ApiResponse(code = 403, message = "authorization error"),
    })
    @PostMapping({"/isAvailable"})
    CompletableFuture<ValidateProjectAvailableResponseDto> isProjectAvailable(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody ValidateProjectAvailableDto validateProjectAvailableDto,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId) {
        return isAvailable(subject, projectId, validateProjectAvailableDto, serviceProviderId);
    }

    @Deprecated
    @ApiOperation(
            value = "Check is subject authorized to change project"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "Subject is authorized"),
            @ApiResponse(code = 400, message = "validation error"),
            @ApiResponse(code = 401, message = "authentication error"),
            @ApiResponse(code = 403, message = "authorization error"),
    })
    @PostMapping("/is-available")
    CompletableFuture<ValidateProjectAvailableResponseDto> isProjectAvailableOld(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody ValidateProjectAvailableDto validateProjectAvailableDto,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId) {
        return isAvailable(subject, projectId, validateProjectAvailableDto, serviceProviderId);
    }

    private CompletableFuture<ValidateProjectAvailableResponseDto> isAvailable(
            AuthSubject subject,
            String projectId,
            ValidateProjectAvailableDto validateProjectAvailableDto,
            String serviceProviderId)
    {
        if (StringUtils.isEmpty(serviceProviderId)) {
            // user token path
            return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
                    .handle((r, throwable) -> resultToResponse(throwable));
        }
        // service provider path
        // try authorize asUser for config editing permission
        return authorizer.authorize(subject, AuthorizationObject.serviceProvider(serviceProviderId, projectId), Permission.ALERT_MANAGEMENT)
                .thenCompose(account -> authorizer.authorize(new AsUserSubject(validateProjectAvailableDto.subjectId), projectId, Permission.CONFIGS_UPDATE)
                        .handle((r, throwable) -> resultToResponse(throwable)));
    }

    private ValidateProjectAvailableResponseDto resultToResponse(Throwable throwable) {
        if (throwable == null) {
            return new ValidateProjectAvailableResponseDto();
        }
        if (Throwables.getRootCause(throwable) instanceof AuthorizationException) {
            var dto = new ValidateProjectAvailableResponseDto();
            dto.configUpdatePermission = false;
            return dto;
        }
        throw new RuntimeException(throwable);
    }

}
