package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.solomon.abc.validator.AbcServiceFieldValidator;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayMigrationConfig;
import ru.yandex.solomon.core.conf.ProjectsManager;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ProjectPermission;
import ru.yandex.solomon.core.exceptions.MethodNotAllowedException;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.gateway.api.v2.dto.FeatureFlagDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.ProjectDto;
import ru.yandex.solomon.gateway.api.v2.validation.ProjectAclValidator;
import ru.yandex.solomon.ydb.page.PageOptions;


/**
 * @author Sergey Polovko
 */
@Api(tags = "projects")
@RestController
@RequestMapping(path = "/api/v2/projects", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ProjectsManager.class, AbcServiceFieldValidator.class })
public class ProjectsController {

    private static final Logger logger = LoggerFactory.getLogger(ProjectsController.class);

    private final ProjectsManager projectsManager;
    private final AbcServiceFieldValidator abcServiceFieldValidator;
    private final ProjectAclValidator projectAclValidator;
    private final FeatureFlagsHolder featureFlagsHolder;
    private final AlertApi alertApi;
    private final Authorizer authorizer;
    private final TGatewayMigrationConfig migrationConfig;
    private final TGatewayCloudConfig cloudConfig;

    @Autowired
    public ProjectsController(
            ProjectsManager projectsManager,
            AbcServiceFieldValidator abcServiceFieldValidator,
            AlertApi alertApi,
            Authorizer authorizer,
            TGatewayConfig gatewayConfig,
            FeatureFlagsHolder featureFlagsHolder)
    {
        this.projectsManager = projectsManager;
        this.abcServiceFieldValidator = abcServiceFieldValidator;
        this.alertApi = alertApi;
        this.authorizer = authorizer;
        this.migrationConfig = gatewayConfig.getGatewayMigrationConfig();
        this.cloudConfig = gatewayConfig.getCloudConfig();
        this.projectAclValidator = new ProjectAclValidator(featureFlagsHolder);
        this.featureFlagsHolder = featureFlagsHolder;
    }

    @ApiOperation(
        value = "list available projects",
        notes = "This action returns all available for current user projects."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
    })
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "text", value = "filter projects by id or name", dataType = "string"),
        @ApiImplicitParam(paramType = "query", name = "filterByPermissions", value = "filter projects by project permissions for current user", dataType = "array"),
        @ApiImplicitParam(paramType = "query", name = "page", value = "page number (starting from 0)", dataType = "integer", defaultValue = "0"),
        @ApiImplicitParam(paramType = "query", name = "pageSize", value = "page size", dataType = "integer", defaultValue = "30"),
    })
    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<Object> allProjects(
        @RequireAuth AuthSubject subject,
        @RequestParam(value = "_usePagination", defaultValue = "false") boolean usePagination,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "filterByAbc", defaultValue = "") String abcFilter,
        @RequestParam(value = "filterByPermissions", required = false) EnumSet<ProjectPermission> permissions,
        PageOptions pageOpts)
    {
        if (permissions == null || permissions.isEmpty()) {
            return fallBackProjectsList(subject, permissions, text, abcFilter, usePagination ? pageOpts: PageOptions.ALL);
        }
        return authorizer.getAvailableAuthorizationObjects(subject, mapPermissions(permissions), EnumSet.of(AuthorizationObject.Type.CLASSIC))
                .thenCompose(resultData -> {
                    if (resultData.objectIds().isEmpty()) {
                        return fallBackProjectsList(subject, permissions, text, abcFilter, usePagination ? pageOpts: PageOptions.ALL);
                    }
                    var projectIds = resultData.getClassicObjects().stream()
                            .map(AuthorizationObject.ClassicAuthorizationObject::projectId)
                            .collect(Collectors.toSet());
                    if (usePagination) {
                        return projectsManager.find(text, abcFilter, projectIds, pageOpts)
                                .thenApply(result -> PagedResultDto.fromModel(result, ProjectDto::fromModel));
                    }
                    logger.warn("request with deprecated _usePagination=false");
                    return projectsManager.find(text, abcFilter, projectIds, PageOptions.ALL)
                            .thenApply(result -> result.getResult().stream()
                                    .map(ProjectDto::fromModel)
                                    .collect(Collectors.toList()));
                });
    }

    private CompletableFuture<Object> fallBackProjectsList(
            AuthSubject subject,
            EnumSet<ProjectPermission> permissions,
            String text,
            String abcFilter,
            PageOptions pageOptions)
    {
        String login = AuthSubject.getLogin(subject)
                .orElse(subject.getUniqueId());
        if (pageOptions != PageOptions.ALL) {
            return projectsManager.find(text, abcFilter, login, permissions, pageOptions)
                    .thenApply(result -> PagedResultDto.fromModel(result, ProjectDto::fromModel));
        }

        logger.warn("request with deprecated _usePagination=false");

        // For backward comparatively
        return projectsManager.find(text, abcFilter, login, permissions, pageOptions)
                .thenApply(result -> result.getResult().stream()
                        .map(ProjectDto::fromModel)
                        .collect(Collectors.toList()));
    }

    private Set<Role> mapPermissions(EnumSet<ProjectPermission> permissions) {
        Set<Role> set = new HashSet<>(permissions.size());
        for (ProjectPermission permission : permissions) {
            switch (permission) {
                case READ -> set.add(Role.VIEWER);
                case CONFIG_UPDATE, CONFIG_DELETE -> {
                    set.add(Role.PROJECT_ADMIN);
                    set.add(Role.EDITOR);
                }
                case WRITE -> {
                    set.add(Role.VIEWER);
                    set.add(Role.PUSHER);
                }
                case UPDATE, ACTIVATE_DEACTIVATE, DELETE -> {
                }
            }
        }
        return set.isEmpty() ? EnumSet.noneOf(Role.class) : EnumSet.copyOf(set);
    }

    @ApiOperation(
        value = "create project",
        notes = "This action will save project document if there is no already existed project with given id."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<ProjectDto> createProject(
        @RequireAuth AuthSubject subject,
        @RequestBody ProjectDto project)
    {
        if (cloudConfig.getForbidProjectCreation()) {
            throw new MethodNotAllowedException(
                    "Manual project creation is forbidden. " +
                    "Please create cloud in resource manager and wait for synchronization.");
        }

        Instant now = Instant.now();
        project.setCreatedAt(now);
        project.setUpdatedAt(now);
        project.setOwner(subject.getUniqueId());
        project.setCreatedBy(subject.getUniqueId());
        project.setUpdatedBy(subject.getUniqueId());
        project.validate();

        projectAclValidator.validateCreate(project);

        return abcServiceFieldValidator.validate(project.getAbcService(), true)
            .thenCompose(unit -> projectsManager.createProject(
                updateCreatingProjectUsingMigrationConfig(ProjectDto.toModel(project))))
            .thenApply(ProjectDto::fromModel);
    }

    private Project updateCreatingProjectUsingMigrationConfig(Project project) {
        Project.Builder builder = project.toBuilder();

        builder.setOnlyAuthPush(true);

        if (migrationConfig.getOnlyMetricNameShardsInNewProjects()) {
            builder.setOnlyMetricNameShards(true);
        }

        if (migrationConfig.getOnlyNewFormatWritesInNewProjects()) {
            builder.setOnlyNewFormatWrites(true);
        }

        if (migrationConfig.getOnlyNewFormatReadsInNewProjects()) {
            builder.setOnlyNewFormatReads(true);
        }

        return builder.build();
    }

    @ApiOperation(
        value = "read one project",
        notes = "This action returns single project found by given id."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "project was not found"),
    })
    @RequestMapping(path = "/{id}", method = RequestMethod.GET)
    CompletableFuture<ProjectDto> getProject(@RequireAuth AuthSubject subject, @PathVariable("id") String id) {
        return authorizer.authorize(subject, id, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> projectsManager.getProject(id))
            .thenApply(ProjectDto::fromModel);
    }

    @ApiOperation(
        value = "update project",
        notes = "This action will update already existed project with given document."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "project was not found"),
    })
    @RequestMapping(path = "/{id}", method = RequestMethod.PUT)
    CompletableFuture<ProjectDto> updateProject(
        @RequireAuth AuthSubject subject,
        @PathVariable("id") String id,
        @RequestBody ProjectDto project)
    {
        project.setId(id);
        project.setUpdatedAt(Instant.now());
        project.setUpdatedBy(subject.getUniqueId());
        project.validate();

        return authorizer.authorize(subject, id, Permission.CONFIGS_UPDATE)
            .thenCompose(account -> abcServiceFieldValidator.validate(project.getAbcService(), false)
                .thenCompose(unit -> {
                    boolean canUpdateAny = account.can(Permission.CONFIGS_UPDATE_ANY);
                    return projectsManager.updateProject(account.getId(), ProjectDto.toModel(project), canUpdateAny, true);
                })
                .thenApply(ProjectDto::fromModel));
    }

    @ApiOperation(
        value = "delete project",
        notes = "This action will delete already existed project."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "project was not found"),
    })
    @RequestMapping(path = "/{id}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteProject(@RequireAuth AuthSubject subject, @PathVariable("id") String id) {
        return authorizer.authorize(subject, id, Permission.CONFIGS_DELETE)
            .thenCompose(account -> {
                // for cloud accounts we check permissions in AccessService instead of
                // relying on permissions stored in project
                boolean canDeleteByNonOwner =
                        account.getAuthType() == AuthType.IAM ||
                        account.can(Permission.CONFIGS_UPDATE_ANY) ||
                        account.can(Permission.PROJECTS_DELETE);
                if (!account.can(Permission.PROJECTS_DELETE)) {
                    if (!SolomonTeam.isMember(subject)) {
                        if (featureFlagsHolder.hasFlag(FeatureFlag.USE_PM_JUGGLER_INTEGRATION, FeatureFlag.USE_PM_JUGGLER_INTEGRATION.name())) {
                            return CompletableFuture.failedFuture(new AuthorizationException("Can't delete project '" + id + "' use ticket for that."));
                        }
                    }
                }
                return projectsManager.deleteProject(account.getId(), id, canDeleteByNonOwner, SolomonTeam.isMember(subject));
            })
            .thenAccept(aVoid -> {
                TDeletionNotificationRequest alertRequest = TDeletionNotificationRequest.newBuilder()
                    .setProjectId(id)
                    .build();

                // we don't want to affect gateway responses if alerting is lagging
                alertApi.notifyOnDeletionProject(alertRequest)
                    .whenComplete((r, t) -> {
                        if (t != null) {
                            ERequestStatusCode statusCode = r.getRequestStatus();
                            logger.info("couldn't delete project:{} from alerting statusCode:{} message:{}",
                                id, statusCode, r.getStatusMessage());
                        }
                    });
            });
    }

    @ApiOperation(
            value = "delete project entities",
            notes = "This action will delete existed project entities."
    )
    @ApiResponses({
            @ApiResponse(code = 401, message = "authentication error"),
            @ApiResponse(code = 403, message = "authorization error"),
            @ApiResponse(code = 404, message = "project was not found"),
    })
    @RequestMapping(path = "/{id}/deleteEntities", method = RequestMethod.POST)
    CompletableFuture<Void> deleteProjectEntities(@RequireAuth AuthSubject subject, @PathVariable("id") String id) {
        return authorizer.authorize(subject, id, Permission.CONFIGS_DELETE)
                .thenCompose(account -> {
                    // for cloud accounts we check permissions in AccessService instead of
                    // relying on permissions stored in project
                    boolean canDeleteByNonOwner =
                            account.getAuthType() == AuthType.IAM ||
                                    account.can(Permission.CONFIGS_UPDATE_ANY);
                    return projectsManager.deleteProjectEntities(account.getId(), id, canDeleteByNonOwner);
                })
                .thenAccept(aVoid -> {
                    TDeletionNotificationRequest alertRequest = TDeletionNotificationRequest.newBuilder()
                            .setProjectId(id)
                            .build();

                    // we don't want to affect gateway responses if alerting is lagging
                    alertApi.notifyOnDeletionProject(alertRequest)
                            .whenComplete((r, t) -> {
                                if (t != null) {
                                    ERequestStatusCode statusCode = r.getRequestStatus();
                                    logger.info("couldn't delete project alerts:{} from alerting statusCode:{} message:{}",
                                            id, statusCode, r.getStatusMessage());
                                }
                            });
                });
    }

    @ApiOperation(
            value = "read project feature-flags",
            notes = "This action returns project feature-flags."
    )
    @ApiResponses({
            @ApiResponse(code = 401, message = "authentication error"),
            @ApiResponse(code = 403, message = "authorization error"),
            @ApiResponse(code = 404, message = "project was not found"),
    })
    @RequestMapping(path = "/{id}/featureFlags", method = RequestMethod.GET)
    CompletableFuture<List<FeatureFlagDto>> getProjectFeatureFlags(
            @RequireAuth AuthSubject subject,
            @PathVariable("id") String id)
    {
        return authorizer.authorize(subject, id, Permission.CONFIGS_GET)
                .thenApply(aVoid -> featureFlagsHolder.flags(id))
                .thenApply(featureFlags -> FeatureFlagDto.from(featureFlags,
                        featureFlag -> featureFlagsHolder.define(featureFlag, id, null, null, null)));
    }
}
