package ru.yandex.solomon.gateway.api.v2;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.google.protobuf.TextFormat;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.auth.http.OptionalAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolverNoOp;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.core.conf.ServicesManager;
import ru.yandex.solomon.core.conf.UnknownShardException;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.gateway.api.v2.dto.PushResultDto;
import ru.yandex.solomon.gateway.push.PushFormats;
import ru.yandex.solomon.gateway.push.PushMetricProcessor;
import ru.yandex.solomon.gateway.push.PushRequest;
import ru.yandex.solomon.gateway.push.PushStatusToHttpStatus;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.selfmon.trace.SpanAwareFuture;
import ru.yandex.solomon.util.SolomonEnv;

import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static java.util.concurrent.CompletableFuture.completedFuture;


/**
 * @author Sergey Polovko
 */
@Api(tags = "push")
@RestController
@RequestMapping(path = "/api/v2/push", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ServicesManager.class, AuthPushMetrics.class})
public class PushController {

    private static final Logger logger = LoggerFactory.getLogger(PushController.class);

    private final PushMetricProcessor pushMetricProcessor;
    private final TGatewayConfig gatewayConfig;
    private final Authorizer authorizer;
    private final FeatureFlagsHolder flags;
    private final AuthPushMetrics authMetrics;
    private final FolderResolver folderResolver;

    @Autowired
    public PushController(
        PushMetricProcessor pushMetricProcessor,
        TGatewayConfig gatewayConfig,
        Authorizer authorizer,
        FeatureFlagsHolder flags,
        AuthPushMetrics authMetrics,
        Optional<FolderResolver> folderResolver)
    {
        this.pushMetricProcessor = pushMetricProcessor;
        this.gatewayConfig = gatewayConfig;
        this.authorizer = authorizer;
        this.flags = flags;
        this.authMetrics = authMetrics;
        this.folderResolver = folderResolver.orElseGet(FolderResolverNoOp::new);
    }

    @ApiOperation(
        value = "Push metrics data",
        notes = "This action pushes data to Solomon (recommended). See https://solomon.yandex-team.ru/docs/operations/metrics/push for more information."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "project", value = "Project to push data", dataType = "string", required = true),
        @ApiImplicitParam(paramType = "query", name = "cluster", value = "Cluster label name to push data", dataType = "string", required = true),
        @ApiImplicitParam(paramType = "query", name = "service", value = "Service label name to push data", dataType = "string", required = true),
        @ApiImplicitParam(paramType = "body", name = "request", value = "Metrics data in JSON or SPACK format. Format determined by Content-Type header"),
    })
    @ApiResponses({
        @ApiResponse(code = 200, message = "success", response = PushResultDto.class),
        @ApiResponse(code = 400, message = "bad request error"),
        @ApiResponse(code = 404, message = "unknown shard error"),
    })
    @RequestMapping(method = RequestMethod.POST)
    @ResponseBody
    public CompletableFuture<ResponseEntity<PushResultDto>> push(
        @OptionalAuth AuthSubject subject,
        @RequestParam("project") String project,
        @RequestParam("cluster") String cluster,
        @RequestParam("service") String service,
        @RequestParam(name = "requestId", required = false) String requestId,
        @RequestBody byte[] body,
        ServerHttpRequest httpRequest)
    {
        // TODO: use requestId to guaranty push idempotency

        boolean isAuthDisabled = gatewayConfig.getDisabledAuthInPushApi();

        // in internal Solomon instance authentication is mandatory
        if (!isAuthDisabled && subject.getAuthType() == AuthType.Anonymous) {
            throw new AuthenticationException("cannot authenticate request", true);
        }

        final String contentType = httpRequest.getHeaders().getFirst(CONTENT_TYPE.toString());
        if (StringUtils.isEmpty(contentType)) {
            throw new BadRequestException(CONTENT_TYPE + " header is missing");
        }

        final MetricFormat format = PushFormats.byContentType(contentType);
        if (format == MetricFormat.METRIC_FORMAT_UNSPECIFIED) {
            String msg = "cannot determine metrics format by given content type: '" + contentType + '\'';
            throw new BadRequestException(msg);
        }

        final ShardKey shardKey;
        try {
            shardKey = new ShardKey(project, cluster, service);
        } catch (Throwable e) {
            String msg = String.format(
                "incomplete values for (project='%s', cluster='%s', service='%s')",
                project, cluster, service);
            throw new BadRequestException(msg);
        }

        return resolveFolderId(shardKey)
            .thenCompose(
                folderId -> doPush(
                    subject,
                    body,
                    isAuthDisabled,
                    format,
                    shardKey,
                    folderId));
    }

    private CompletableFuture<ResponseEntity<PushResultDto>> doPush(
        AuthSubject subject,
        byte[] body,
        boolean isAuthDisabled,
        MetricFormat format,
        ShardKey shardKey,
        String folderId)
    {
        var project = shardKey.getProject();
        var cluster = shardKey.getCluster();
        var service = shardKey.getService();

        final CompletableFuture<Account> authorizedAccount;
        if (isAuthDisabled && !flags.hasFlag(FeatureFlag.AUTHORIZE_ON_PUSH_WHEN_GLOBALLY_DISABLED, project, cluster, service)) {
            // XXX: temporary hack to disable auth while we are working on
            //      updating cloud tokens
            String id = SolomonEnv.PRODUCTION.isActive() ? "aje40000000000000016" : "bfb40000000000000016";
            Account account = new Account(id, AuthType.IAM, AuthorizationType.IAM, RoleSet.PROJECT_VIEW_AND_PUSH);
            authorizedAccount = completedFuture(account);
            if (flags.hasFlag(FeatureFlag.BACKGROUND_CHECK_AUTH_ON_PUSH_WHEN_GLOBALLY_DISABLED, project, cluster, service)) {
                authMetrics.forFuture(shardKey, authorizer.authorize(subject, project, folderId, Permission.DATA_WRITE));
            }
        } else {
            // in internal Solomon instance perform authorization process as usual
            authorizedAccount = authorizer.authorize(subject, project, folderId, Permission.DATA_WRITE);
        }

        final ByteBuf content = Unpooled.wrappedBuffer(body);
        return SpanAwareFuture.wrap(authorizedAccount)
            .thenCompose(account -> {
                var request = new PushRequest(shardKey, format, content, body.length, account);
                return pushMetricProcessor.processPush(request, false)
                        .whenComplete((r, t) -> logResult(shardKey, r, t));
            })
            .thenApply(r -> {
                HttpStatus httpStatus = PushStatusToHttpStatus.convert(r.getStatus());
                return ResponseEntity.status(httpStatus)
                    .body(PushResultDto.fromModel(r));
            });
    }

    private CompletableFuture<String> resolveFolderId(ShardKey shardKey) {
        return folderResolver.resolveFolderId(shardKey);
    }

    private void logResult(ShardKey shardKey, TDataProcessResponse r, Throwable t) {
        if (t != null) {
            Throwable cause = CompletableFutures.unwrapCompletionException(t);
            if (cause instanceof UnknownShardException) {
                logger.warn("push into unknown shard {}", shardKey);
            } else {
                logger.warn("push into {} failed", shardKey, t);
            }
        } else {
            logger.info("push into {} processed: {}", shardKey, TextFormat.shortDebugString(r));
        }
    }
}
