package ru.yandex.solomon.gateway.api.v2;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.swagger.annotations.ApiOperation;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.RoleListRequest;
import ru.yandex.monitoring.api.v3.UpdateRoleListRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.v3.intranet.ProjectRoleService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Alexey Trushkin
 */
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/roles", produces = MediaType.APPLICATION_JSON_VALUE)
public class RolesPublicController {

    private final ProjectRoleService projectRoleService;

    public RolesPublicController(Optional<ProjectRoleService> projectRoleService) {
        this.projectRoleService = projectRoleService.orElse(null);
    }

    @ApiOperation(value = "List project roles")
    @GetMapping
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        RoleListRequest listRequest = RoleListRequest.newBuilder()
                .setProjectId(projectId)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return projectRoleService.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @ApiOperation(value = "Update project roles. Can add and delete roles")
    @PostMapping
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody byte[] body)
    {
        UpdateRoleListRequest.Builder builder = UpdateRoleListRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        return projectRoleService.update(builder.build(), subject).thenApply(empty -> null);
    }

    @ApiOperation(value = "Check project roles update status")
    @PostMapping(path = "/status")
    CompletableFuture<String> status(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        UpdateRoleListRequest.Builder builder = UpdateRoleListRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        return projectRoleService.status(builder.build(), subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
