package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ServicesManager;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterServiceAssociationDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.ServiceDto;
import ru.yandex.solomon.gateway.api.v2.dto.ServiceListItemDto;
import ru.yandex.solomon.gateway.api.v2.dto.ShardListItemDto;
import ru.yandex.solomon.ydb.page.PageOptions;


/**
 * @author Sergey Polovko
 */
@Api(tags = "services")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/services", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ServicesManager.class })
public class ServicesController {

    @Autowired
    private ServicesManager servicesManager;
    @Autowired
    private Authorizer authorizer;


    @ApiOperation(
        value = "list available services",
        notes = "This action returns project's services if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "page", value = "page number (starting from 0)", dataType = "integer", defaultValue = "0"),
        @ApiImplicitParam(paramType = "query", name = "pageSize", value = "page size", dataType = "integer", defaultValue = "30"),
    })
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<ServiceListItemDto>> getAll(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "monitoringModel", required = false, defaultValue = "UNSPECIFIED") ShardSettings.Type monitoringModel,
        PageOptions pageOptions)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
            .thenCompose(aVoid -> servicesManager.getServices(projectId, "", pageOptions, text, monitoringModel))
            .thenApply(services -> PagedResultDto.fromModel(services, ServiceListItemDto::fromModel));
    }

    @ApiOperation(
        value = "create service",
        notes = "This action will save service document if there is no already existed service with given id."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<ServiceDto> createService(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestBody ServiceDto service)
    {
        Instant now = Instant.now();
        service.setCreatedAt(now);
        service.setUpdatedAt(now);
        service.setCreatedBy(subject.getUniqueId());
        service.setUpdatedBy(subject.getUniqueId());
        service.setProjectId(projectId);
        service.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_CREATE)
            .thenCompose(aVoid -> servicesManager.createService(ServiceDto.toModel(service)))
            .thenApply(ServiceDto::fromModel);
    }

    @ApiOperation(
        value = "read one service",
        notes = "This action returns single project's service found by given serviceId."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "service was not found"),
    })
    @RequestMapping(path = "/{serviceId}", method = RequestMethod.GET)
    CompletableFuture<ServiceDto> getService(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("serviceId") String serviceId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> servicesManager.getService(projectId, "", serviceId))
            .thenApply(ServiceDto::fromModel);
    }

    @ApiOperation(
        value = "update service",
        notes = "This action will update already existed project's service with given document."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "service was not found"),
    })
    @RequestMapping(path = "/{serviceId}", method = RequestMethod.PUT)
    CompletableFuture<ServiceDto> updateService(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("serviceId") String serviceId,
        @RequestBody ServiceDto service)
    {
        Instant now = Instant.now();
        service.setUpdatedAt(now);
        service.setUpdatedBy(subject.getUniqueId());
        service.setId(serviceId);
        service.setProjectId(projectId);
        service.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
            .thenCompose(account -> {
                return servicesManager.updateService(ServiceDto.toModel(service));
            })
            .thenApply(ServiceDto::fromModel);
    }

    @ApiOperation(
        value = "delete service",
        notes = "This action will delete already existed service."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "service was not found"),
    })
    @RequestMapping(path = "/{serviceId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteService(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("serviceId") String serviceId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_DELETE)
            .thenCompose(aVoid -> servicesManager.deleteService(projectId, "", serviceId));
    }

    @ApiOperation(
        value = "associated clusters",
        notes = "This action returns clusters associated with service found by given serviceId."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(path = "/{serviceId}/clusters", method = RequestMethod.GET)
    CompletableFuture<List<ClusterServiceAssociationDto>> getServiceClusters(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("serviceId") String serviceId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> servicesManager.getServiceAssociations(projectId, "", serviceId))
            .thenApply(associations -> associations.stream()
                .map(ClusterServiceAssociationDto::fromModel)
                .collect(Collectors.toList()));
    }

    @ApiOperation(
            value = "associated shards",
            notes = "This action returns shards associated with service found by given serviceId."
    )
    @ApiResponses({
            @ApiResponse(code = 401, message = "authentication error"),
            @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(path = "/{serviceId}/shards", method = RequestMethod.GET)
    CompletableFuture<List<ShardListItemDto>> getServiceShards(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("serviceId") String serviceId,
            @RequestParam(value = "state", required = false) EnumSet<ShardState> state)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                .thenCompose(aVoid -> servicesManager.getServiceShards(projectId, "", serviceId, state))
                .thenApply(shards -> shards.stream()
                        .map(ShardListItemDto::fromModel)
                        .collect(Collectors.toList()));
    }
}
