package ru.yandex.solomon.gateway.api.v2;

import java.time.Instant;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ShardsManager;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.gateway.api.v2.dto.FeatureFlagDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.ShardDto;
import ru.yandex.solomon.gateway.api.v2.dto.ShardListItemDto;
import ru.yandex.solomon.gateway.api.v2.dto.ShardTargetStatusDto;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.net.KnownDc;
import ru.yandex.solomon.ydb.page.PageOptions;

/**
 * @author Sergey Polovko
 */
@Api(tags = "shards")
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/shards", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ShardsManager.class })
public class ShardsController {

    @Autowired
    private ShardsManager shardsManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private CoremonClient coremonClient;
    @Autowired
    private FeatureFlagsHolder featureFlagsHolder;

    @ApiOperation(
        value = "list available shards",
        notes = "This action returns project's shards if user have permissions to read that project."
    )
    @ApiImplicitParams({
        @ApiImplicitParam(paramType = "query", name = "page", value = "page number (starting from 0)", dataType = "integer", defaultValue = "0"),
        @ApiImplicitParam(paramType = "query", name = "pageSize", value = "page size", dataType = "integer", defaultValue = "30"),
    })
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<ShardListItemDto>> getAll(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "text", defaultValue = "") String text,
            @RequestParam(value = "fullModel", defaultValue = "false") boolean fullModel,
            @RequestParam(value = "state", required = false) EnumSet<ShardState> state,
            PageOptions pageOptions)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
            .thenCompose(aVoid -> shardsManager.getProjectShards(projectId, "", pageOptions, state, text, fullModel))
            .thenApply(shards -> PagedResultDto.fromModel(shards, ShardListItemDto::fromModel));
    }

    @ApiOperation(
        value = "create shard",
        notes = "This action will save shard document if there is no already existed shard with given id."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
    })
    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<ShardDto> createShard(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "fullModel", defaultValue = "false") boolean fullModel,
        @RequestBody ShardDto shard)
    {
        Instant now = Instant.now();
        shard.setCreatedAt(now);
        shard.setUpdatedAt(now);
        shard.setCreatedBy(subject.getUniqueId());
        shard.setUpdatedBy(subject.getUniqueId());
        shard.setProjectId(projectId);
        shard.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_CREATE)
            .thenCompose(account -> {
                boolean canUpdateInternals = account.can(Permission.CONFIGS_UPDATE_INTERNALS);
                return shardsManager.getClusterServiceNames(projectId, "", shard.getClusterId(), shard.getServiceId())
                    .thenCompose(names -> shardsManager.createShard(ShardDto.toModel(shard, names), canUpdateInternals, fullModel))
                    .thenApply(s -> ShardDto.fromModel(s, coremonClient));
            });
    }

    @ApiOperation(
        value = "read one shard",
        notes = "This action returns single project's shard found by given shardId."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "shard was not found"),
    })
    @RequestMapping(path = "/{shardId}", method = RequestMethod.GET)
    CompletableFuture<ShardDto> getShard(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("shardId") String shardId,
        @RequestParam(value = "fullModel", defaultValue = "false") boolean fullModel)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> shardsManager.getShard(projectId, "", shardId, fullModel))
            .thenApply(s -> ShardDto.fromModel(s, coremonClient));
    }

    @ApiOperation(
        value = "update shard",
        notes = "This action will update already existed project's shard with given document."
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "shard was not found"),
    })
    @RequestMapping(path = "/{shardId}", method = RequestMethod.PUT)
    CompletableFuture<ShardDto> updateShard(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("shardId") String shardId,
        @RequestBody ShardDto shard,
        @RequestParam(value = "fullModel", defaultValue = "false") boolean fullModel)
    {
        Instant now = Instant.now();
        shard.setUpdatedAt(now);
        shard.setUpdatedBy(subject.getUniqueId());
        shard.setId(shardId);
        shard.setProjectId(projectId);
        shard.validate();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
            .thenCompose(account -> {
                boolean canUpdateInternals = account.can(Permission.CONFIGS_UPDATE_INTERNALS);
                return shardsManager.getClusterServiceNames(projectId, "", shard.getClusterId(), shard.getServiceId())
                    .thenCompose(names -> shardsManager.updateShard(ShardDto.toModel(shard, names), canUpdateInternals, fullModel))
                    .thenApply(s -> ShardDto.fromModel(s, coremonClient));
            });
    }

    @ApiOperation(
        value = "delete shard",
        notes = "This action will delete already existed shard."
    )
    @ApiResponses({
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "shard was not found"),
    })
    @RequestMapping(path = "/{shardId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteShard(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("shardId") String shardId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_DELETE)
            .thenCompose(aVoid -> shardsManager.deleteShard(projectId, "", shardId));
    }

    @ApiOperation(
        value = "get shard targets statuses",
        notes = "This action will retrieve paged list of statuses of pulled shard targets by particular Solomon Fetcher host.",
        response = ShardTargetStatusDtoPage.class
    )
    @ApiResponses({
        @ApiResponse(code = 400, message = "validation error"),
        @ApiResponse(code = 401, message = "authentication error"),
        @ApiResponse(code = 403, message = "authorization error"),
        @ApiResponse(code = 404, message = "shard was not found"),
    })
    @RequestMapping(path = "/{shardId}/targets", method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<ShardTargetStatusDto>> targetsStatus(
        @RequireAuth AuthSubject subject,
        @PathVariable("projectId") String projectId,
        @PathVariable("shardId") String shardId,
        @RequestParam(name = "fetcherHost", defaultValue = "") String fetcherHost,
        @RequestParam(name = "hostGlob", defaultValue = "") String hostGlob,
        @RequestParam(name = "dc", required = false) @Nullable KnownDc dc,
        @RequestParam(name = "status", required = false) @Nullable String statusStr,
        PageOptions pageOptions)
    {
        boolean notOkStatus = "NOT_OK".equals(statusStr);
        UrlStatusType status = (notOkStatus || StringUtils.isEmpty(statusStr))
            ? null
            : UrlStatusType.valueOf(statusStr);

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> shardsManager.fetcherTargetsStatus(projectId, "", shardId, fetcherHost, hostGlob, dc, status, notOkStatus, pageOptions))
            .thenApply(targetsStatus -> PagedResultDto.fromModel(targetsStatus, ShardTargetStatusDto::fromModel));
    }

    @ApiOperation(
            value = "read shard feature-flags",
            notes = "This action returns shard feature-flags."
    )
    @ApiResponses({
            @ApiResponse(code = 401, message = "authentication error"),
            @ApiResponse(code = 403, message = "authorization error"),
            @ApiResponse(code = 404, message = "project was not found"),
    })
    @RequestMapping(path = "/{shardId}/featureFlags", method = RequestMethod.GET)
    CompletableFuture<List<FeatureFlagDto>> getProjectFeatureFlags(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("shardId") String shardId)
    {
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                .thenCompose(aVoid -> shardsManager.getShard(projectId, "", shardId, false))
                .thenApply(shardExtended -> {
                    var shard = shardExtended.shard();
                    var flags = featureFlagsHolder.flags(projectId);
                    return FeatureFlagDto.from(flags,
                            featureFlag -> featureFlagsHolder.define(featureFlag, projectId, shard.getId(), shard.getClusterId(), shard.getServiceId()));
                });
    }

    // used only for swagger documentation generation
    private static final class ShardTargetStatusDtoPage extends PagedResultDto<ShardTargetStatusDto> {}
}
