package ru.yandex.solomon.gateway.api.v2.dto;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonValue;

import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.ProjectPermission;
import ru.yandex.solomon.core.exceptions.BadRequestException;


/**
 * @author Sergey Polovko
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class AclDto {

    private final List<AclEntryDto> entries;

    @JsonValue
    public List<AclEntryDto> getEntries() {
        return entries;
    }

    @JsonCreator
    public AclDto(@Nullable List<AclEntryDto> entries) {
        this.entries = entries;
    }

    @JsonIgnore
    public boolean isEmpty() {
        return entries == null || entries.isEmpty();
    }

    public void validate() {
        if (entries != null) {
            for (AclEntryDto entry : entries) {
                if (entry == null) {
                    throw new BadRequestException("nullable acl entry");
                }

                entry.validate();
            }
        }
    }

    @Nonnull
    public static AclDto fromModel(@Nonnull Acl acl) {
        HashMap<String, AclEntryDto> aclEntryByUser = new HashMap<>();

        acl.getCanRead().forEach(user -> setPermission(aclEntryByUser, user, ProjectPermission.READ));
        acl.getCanUpdate().forEach(user -> setPermission(aclEntryByUser, user, ProjectPermission.CONFIG_UPDATE));
        acl.getCanDelete().forEach(user -> setPermission(aclEntryByUser, user, ProjectPermission.CONFIG_DELETE));
        acl.getCanWrite().forEach(user -> setPermission(aclEntryByUser, user, ProjectPermission.WRITE));

        return new AclDto(new ArrayList<>(aclEntryByUser.values()));
    }

    private static void setPermission(
        @Nonnull HashMap<String, AclEntryDto> aclEntryByUser,
        @Nonnull String user,
        @Nonnull ProjectPermission permission)
    {
        AclEntryDto aclEntryDto = aclEntryByUser.computeIfAbsent(user, s -> {
            AclEntryDto newAclEntryDto = new AclEntryDto();
            newAclEntryDto.setLogin(s);
            newAclEntryDto.setPermissions(EnumSet.noneOf(ProjectPermission.class));
            return newAclEntryDto;
        });

        aclEntryDto.getPermissions().add(permission);
    }

    @Nonnull
    public static Acl toModel(AclDto dto) {
        if (dto == null || dto.isEmpty()) {
            return Acl.empty();
        }

        HashSet<String> canRead = new HashSet<>();
        HashSet<String> canUpdate = new HashSet<>();
        HashSet<String> canDelete = new HashSet<>();
        HashSet<String> canWrite = new HashSet<>();

        for (AclEntryDto entry : dto.getEntries()) {
            if (entry == null) {
                throw new RuntimeException("Nullable ACL entry in list: " + dto.getEntries().toString());
            }
            String user = entry.getLogin();
            EnumSet<ProjectPermission> permissions = entry.getPermissions();
            if (user == null || permissions == null) {
                throw new RuntimeException("Wrong ACL entry: " + entry.toString());
            }
            if (permissions.contains(ProjectPermission.READ)) {
                canRead.add(user);
            }
            if (permissions.contains(ProjectPermission.CONFIG_UPDATE) || permissions.contains(ProjectPermission.UPDATE)) {
                canUpdate.add(user);
            }
            if (permissions.contains(ProjectPermission.CONFIG_DELETE) || permissions.contains(ProjectPermission.DELETE)) {
                canDelete.add(user);
            }
            if (permissions.contains(ProjectPermission.WRITE)) {
                canWrite.add(user);
            }
        }

        return Acl.of(canRead, canUpdate, canDelete, canWrite);
    }
}
